<?php

namespace App\Exports;

use App\Models\Article;
use Maatwebsite\Excel\Concerns\FromArray;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use Maatwebsite\Excel\Concerns\ShouldAutoSize;

class ClientHistoryExport implements FromArray, WithHeadings, WithMapping, WithTitle, WithColumnWidths, WithEvents, ShouldAutoSize
{
    protected $orders;

    public function __construct($orders)
    {
        $this->orders = $orders;
    }

    /**
     * @return array
     */
    public function array(): array
    {
        $items = [];
        foreach ($this->orders as $order) {
            foreach ($order->orderItems as $orderItem) {
                // Buscar información del artículo
                $article = Article::where('mocaco', $orderItem->mocaco)->first();

                $items[] = [
                    'order' => $order,
                    'orderItem' => $orderItem,
                    'article' => $article
                ];
            }
        }
        return $items;
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            'ID Pedido',
            'Fecha Pedido',
            'Código de Barras',
            'MOCACO',
            'Número de Caja',
            'Familia Artículo',
            'Cadena',
            'Mercado Origen',
            'Precio PVP Max Temporada',
            'Partida Arancelaria',
            'Composition',
            'Campaña',
            'Peso Unitario',
            'Grupo Arancelario',
        ];
    }

    /**
     * @param array $row
     * @return array
     */
    public function map($row): array
    {
        $order = $row['order'];
        $item = $row['orderItem'];
        $article = $row['article'];

        return [
            $order->id,
            $order->order_date ? $order->order_date->format('d/m/Y H:i') : '',
            $item->full_barcode,
            $item->mocaco,
            $item->n_carton,
            // Campos adicionales del artículo
            $article->familia ?? 'N/A',
            $article->cadena ?? 'N/A',
            $article->mercado_origen_articulo ?? 'N/A',
            $article->precio_pvp_maximo_temporada ?? 'N/A',
            $article->partida_arancelaria ?? 'N/A',
            $article->composition ?? 'N/A',
            $article->campana ?? 'N/A',
            $article->peso_unitario ?? 'N/A',
            $article->grupo_arancelario ?? 'N/A',
        ];
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return 'Historial Cliente';
    }

    /**
     * @return array
     */
    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function (AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();

                // Estilo para los headers de la tabla (ahora son 14 columnas: A-N)
                $sheet->getStyle('A1:N1')->getFont()->setBold(true)->setSize(12);
                $sheet->getStyle('A1:N1')->getFill()
                    ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
                    ->setStartColor(new \PhpOffice\PhpSpreadsheet\Style\Color(\PhpOffice\PhpSpreadsheet\Style\Color::COLOR_YELLOW));
            },
        ];
    }

    /**
     * @return array
     */
    public function columnWidths(): array
    {
        return [
            'A' => 12,  // ID Pedido
            'B' => 18,  // Fecha Pedido
            'C' => 20,  // Código de Barras
            'D' => 15,  // MOCACO
            'E' => 15,  // Número de Caja
            'F' => 20,  // Familia Artículo
            'G' => 15,  // Cadena
            'H' => 15,  // Mercado Origen
            'I' => 12,  // Precio PVP
            'J' => 18,  // Partida Arancelaria
            'K' => 30,  // Composition
            'L' => 12,  // Campaña
            'M' => 12,  // Peso Unitario
            'N' => 18,  // Grupo Arancelario
        ];
    }
}
