<?php

namespace App\Exports;

use App\Models\Inventory;
use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Illuminate\Support\Facades\DB;

class InventoryGroupedExport implements FromCollection, WithHeadings, WithMapping, WithTitle, WithColumnWidths
{
    protected $filters;

    /**
     * Constructor to accept filters
     */
    public function __construct(array $filters = [])
    {
        $this->filters = $filters;
    }

    /**
     * Agrupar inventario por campos del artículo y sumar unidades
     */
    public function collection()
    {
        $query = Inventory::query();

        // Ocultar ítems despachados
        $query->where(function ($q) {
            $q->whereNull('status')
                ->orWhereNotIn('status', ['dispatched', 'despachado']);
        });

        // Apply filters
        if (!empty($this->filters['search'])) {
            $search = $this->filters['search'];
            $query->where(function ($q) use ($search) {
                $q->where('full_barcode', 'like', "%{$search}%")
                    ->orWhere('mocaco', 'like', "%{$search}%")
                    ->orWhere('n_carton', 'like', "%{$search}%")
                    ->orWhere('ubicacion', 'like', "%{$search}%");
            });
        }

        if (!empty($this->filters['box_number'])) {
            $query->where('n_carton', 'like', "%{$this->filters['box_number']}%");
        }

        if (!empty($this->filters['location'])) {
            $query->where('ubicacion', 'like', "%{$this->filters['location']}%");
        }

        // Multiselect filter for containers
        if (!empty($this->filters['containers']) && is_array($this->filters['containers'])) {
            $query->whereIn('conteneur', $this->filters['containers']);
        }

        // Multiselect filter for seasons
        if (!empty($this->filters['seasons']) && is_array($this->filters['seasons'])) {
            $query->whereIn('season_int', $this->filters['seasons']);
        }

        // Brand filter
        if (!empty($this->filters['marcas'])) {
            $query->where('cadena', 'like', "%{$this->filters['marcas']}%");
        }

        // Needs review filter
        if (!empty($this->filters['needs_review']) && $this->filters['needs_review']) {
            $query->where('needs_review', 1);
        }

        return $query->select(
            'mocaco',
            'n_carton',
            'conteneur',
            'ubicacion',
            'seccion',
            'familia_articulo_description',
            'cadena',
            'mercado_origen_articulo',
            'precio_pvp_maximo_temporada',
            'partida_arancelaria',
            'composition',
            'campana',
            'peso_unitario',
            'grupo_arancelario',
            'season_int',
            'categoria_seleccionada',
            'famillie_usuario',
            'detail_usuario',
            DB::raw('COUNT(*) as unidades')
        )
            ->groupBy(
                'mocaco',
                'n_carton',
                'conteneur',
                'ubicacion',
                'seccion',
                'familia_articulo_description',
                'cadena',
                'mercado_origen_articulo',
                'precio_pvp_maximo_temporada',
                'partida_arancelaria',
                'composition',
                'campana',
                'peso_unitario',
                'grupo_arancelario',
                'season_int',
                'categoria_seleccionada',
                'famillie_usuario',
                'detail_usuario'
            )
            ->orderBy('mocaco')
            ->orderBy('conteneur')
            ->get();
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        return [
            'MOCACO',
            'Nº Cartón',
            'Contenedor',
            'Ubicación',
            'Sección',
            'Familia Artículo',
            'Cadena',
            'Mercado Origen',
            'Precio PVP Max Temporada',
            'Partida Arancelaria',
            'Composition',
            'Campaña',
            'Peso Unitario',
            'Grupo Arancelario',
            'Season INT',
            'Categoría Seleccionada',
            'Familia Usuario',
            'Detail Usuario',
            'UNIDADES',
        ];
    }

    /**
     * @param mixed $item
     * @return array
     */
    public function map($item): array
    {
        return [
            $item->mocaco,
            $item->n_carton ?? '',
            $item->conteneur ?? '',
            $item->ubicacion ?? '',
            $item->seccion ?? '',
            $item->familia_articulo_description ?? '',
            $item->cadena ?? '',
            $item->mercado_origen_articulo ?? '',
            $item->precio_pvp_maximo_temporada ?? 0,
            $item->partida_arancelaria ?? '',
            $item->composition ?? '',
            $item->campana ?? '',
            $item->peso_unitario ?? 0,
            $item->grupo_arancelario ?? '',
            $item->season_int ?? '',
            $item->categoria_seleccionada ?? '',
            $item->famillie_usuario ?? '',
            $item->detail_usuario ?? '',
            $item->unidades,
        ];
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return 'Inventario Agrupado';
    }

    /**
     * @return array
     */
    public function columnWidths(): array
    {
        return [
            'A' => 15,  // MOCACO
            'B' => 15,  // Nº Cartón
            'C' => 15,  // Contenedor
            'D' => 15,  // Ubicación
            'E' => 15,  // Sección
            'F' => 25,  // Familia
            'G' => 15,  // Cadena
            'H' => 20,  // Mercado Origen
            'I' => 20,  // Precio
            'J' => 20,  // Partida
            'K' => 30,  // Composition
            'L' => 15,  // Campaña
            'M' => 15,  // Peso
            'N' => 20,  // Grupo
            'O' => 15,  // Season
            'P' => 20,  // Categoría
            'Q' => 20,  // Familia Usuario
            'R' => 25,  // Detail Usuario
            'S' => 12,  // UNIDADES
        ];
    }
}
