<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithMapping;
use Maatwebsite\Excel\Concerns\WithTitle;
use Maatwebsite\Excel\Concerns\WithColumnWidths;
use Maatwebsite\Excel\Concerns\WithCustomStartCell;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use App\Models\Order;

class OrderDetailExport implements FromCollection, WithHeadings, WithMapping, WithTitle, WithColumnWidths, WithCustomStartCell, WithEvents
{
    protected $order;

    public function __construct(Order $order)
    {
        $this->order = $order->load(['client', 'user', 'orderItems.inventory.article']);
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    public function collection()
    {
        return $this->order->orderItems;
    }

    /**
     * @return array
     */
    public function headings(): array
    {
        // Alinear columnas con OrdersListExport
        return [
            'ID Pedido',
            'Cliente',
            'Email Cliente',
            'Teléfono',
            'Operario Pedido',
            'Usuario',
            'Status Pedido',
            'Fecha Pedido',
            'Fecha Despacho',
            'Notas Pedido',
            'Código de Barras',
            'MOCACO',
            'Nº Carton',
            'Ubicación',
            'Contenedor',
            'Operario Item',
            'Season INT',
            'Categoría',
            'Familia Usuario',
            'Detail Usuario',
            'Sección',
            'Familia Artículo',
            'Cadena',
            'Mercado Origen',
            'Precio PVP Max',
            'Partida Arancelaria',
            'Composition',
            'Campaña',
            'Peso Unitario',
            'Grupo Arancelario',
            'Status Item',
            'Order ID Item',
        ];
    }

    /**
     * @return string
     */
    public function startCell(): string
    {
        return 'A11'; // Empezar en la fila 11 para dejar espacio para la información del pedido
    }

    /**
     * @param mixed $item
     * @return array
     */
    public function map($item): array
    {
        $order = $this->order;
        $inventory = $item->inventory;
        
        return [
            // Pedido
            $order->id,
            $order->client->client_name ?? '',
            $order->client->email ?? '',
            $order->client->phone ?? '',
            $order->operario_id ?? '',
            $order->user->name ?? '',
            ucfirst($order->status),
            $order->order_date ? $order->order_date->format('d/m/Y H:i') : '',
            $order->dispatched_date ? $order->dispatched_date->format('d/m/Y H:i') : '',
            $order->notes ?? '',
            // Item / Inventario
            $inventory ? ($inventory->full_barcode ?? $item->full_barcode ?? '') : ($item->full_barcode ?? ''),
            $inventory ? ($inventory->mocaco ?? $item->mocaco ?? '') : ($item->mocaco ?? ''),
            $inventory ? ($inventory->n_carton ?? $item->n_carton ?? '') : ($item->n_carton ?? ''),
            $inventory ? $inventory->ubicacion ?? '' : '',
            $inventory ? $inventory->conteneur ?? '' : '',
            $inventory ? $inventory->n_id_operario ?? '' : '',
            $inventory ? $inventory->season_int ?? '' : '',
            $inventory ? $inventory->categoria_seleccionada ?? '' : '',
            $inventory ? $inventory->famillie_usuario ?? '' : '',
            $inventory ? $inventory->detail_usuario ?? '' : '',
            $inventory ? $inventory->seccion ?? '' : '',
            $inventory ? $inventory->familia_articulo_description ?? '' : '',
            $inventory ? $inventory->cadena ?? '' : '',
            $inventory ? $inventory->mercado_origen_articulo ?? '' : '',
            $inventory ? $inventory->precio_pvp_maximo_temporada ?? '' : '',
            $inventory ? $inventory->partida_arancelaria ?? '' : '',
            $inventory ? $inventory->composition ?? '' : '',
            $inventory ? $inventory->campana ?? '' : '',
            $inventory ? $inventory->peso_unitario ?? '' : '',
            $inventory ? $inventory->grupo_arancelario ?? '' : '',
            $inventory ? ($inventory->status ?? 'disponible') : '',
            $inventory ? $inventory->order_id ?? '' : '',
        ];
    }

    /**
     * @return string
     */
    public function title(): string
    {
        return 'Pedido #' . $this->order->id;
    }

    /**
     * @return array
     */
    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function(AfterSheet $event) {
                $sheet = $event->sheet->getDelegate();
                
                // Agregar información del pedido en las primeras filas
                $sheet->setCellValue('A1', 'INFORMACIÓN DEL PEDIDO');
                $sheet->setCellValue('A2', 'ID Pedido:');
                $sheet->setCellValue('B2', $this->order->id);
                $sheet->setCellValue('A3', 'Cliente:');
                $sheet->setCellValue('B3', $this->order->client->client_name ?? 'N/A');
                $sheet->setCellValue('A4', 'Email:');
                $sheet->setCellValue('B4', $this->order->client->email ?? '');
                $sheet->setCellValue('A5', 'Teléfono:');
                $sheet->setCellValue('B5', $this->order->client->phone ?? '');
                $sheet->setCellValue('A6', 'Estado:');
                $sheet->setCellValue('B6', ucfirst($this->order->status));
                $sheet->setCellValue('A7', 'Fecha Pedido:');
                $sheet->setCellValue('B7', $this->order->order_date ? $this->order->order_date->format('d/m/Y H:i') : '');
                $sheet->setCellValue('A8', 'Total Items:');
                $sheet->setCellValue('B8', $this->order->orderItems->count());

                // Estilos para la sección de información
                $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
                $sheet->getStyle('A2:A9')->getFont()->setBold(true);
                
                // Estilo para los headers de la tabla (A11:AF11)
                $sheet->getStyle('A11:AF11')->getFont()->setBold(true)->setSize(12);
                $sheet->getStyle('A11:AF11')->getFill()
                    ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
                    ->setStartColor(new \PhpOffice\PhpSpreadsheet\Style\Color(\PhpOffice\PhpSpreadsheet\Style\Color::COLOR_YELLOW));
            },
        ];
    }

    /**
     * @return array
     */
    public function columnWidths(): array
    {
        return [
            'A' => 12, // ID Pedido
            'B' => 25, // Cliente
            'C' => 25, // Email Cliente
            'D' => 15, // Teléfono
            'E' => 12, // Operario
            'F' => 15, // Usuario
            'G' => 12, // Status
            'H' => 18, // Fecha Pedido
            'I' => 18, // Fecha Despacho
            'J' => 40, // Notas
            'K' => 20, // Código de Barras
            'L' => 15, // MOCACO
            'M' => 12, // Nº Carton
            'N' => 15, // Ubicación
            'O' => 15, // Contenedor
            'P' => 12, // Operario
            'Q' => 12, // Season INT
            'R' => 15, // Categoría
            'S' => 20, // Familia Usuario
            'T' => 20, // Detail Usuario
            'U' => 12, // Sección
            'V' => 20, // Familia Artículo
            'W' => 15, // Cadena
            'X' => 18, // Mercado Origen
            'Y' => 15, // Precio PVP Max
            'Z' => 18, // Partida Arancelaria
            'AA' => 20, // Composition
            'AB' => 15, // Campaña
            'AC' => 15, // Peso Unitario
            'AD' => 18, // Grupo Arancelario
            'AE' => 12, // Status
            'AF' => 12, // Order ID
        ];
    }
}

