<?php

namespace App\Http\Controllers;

use App\Models\Article;
use App\Models\AuditLog;
use App\Traits\AuthorizesResourceAccess;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class ArticleController extends Controller
{
    use AuthorizesResourceAccess;
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $this->authorizeAction('view');

        $query = Article::query();

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('mocaco', 'like', "%{$search}%")
                    ->orWhere('familia', 'like', "%{$search}%")
                    ->orWhere('cadena', 'like', "%{$search}%")
                    ->orWhere('composition', 'like', "%{$search}%");
            });
        }

        if ($request->filled('family')) {
            $query->where('familia', 'like', "%{$request->family}%");
        }

        if ($request->filled('brand')) {
            $query->where('cadena', 'like', "%{$request->brand}%");
        }

        $perPage = $request->get('per_page', 20);
        $articles = $query->orderBy('mocaco')->paginate($perPage)->appends($request->except('page'));

        return view('articles.index', compact('articles'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $this->authorizeAction('create');

        return view('articles.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->authorizeAction('create');

        $request->validate([
            'mocaco' => 'required|string|max:20',
            'seccion' => 'nullable|string|max:50',
            'familia' => 'nullable|string|max:100',
            'cadena' => 'nullable|string|max:50',
            'mercado_origen_articulo' => 'nullable|string|max:100',
            'precio_pvp_maximo_temporada' => 'nullable|numeric',
            'partida_arancelaria' => 'nullable|string|max:50',
            'composition' => 'nullable|string',
            'campana' => 'nullable|string|max:50',
            'peso_unitario' => 'nullable|numeric',
            'grupo_arancelario' => 'nullable|string|max:50',
        ]);

        // Verificar si ya existe un artículo con la misma combinación MOCACO + CAMPANA
        $existing = Article::where('mocaco', $request->mocaco)
            ->where('campana', $request->campana)
            ->first();

        if ($existing) {
            return back()
                ->withInput()
                ->with('error', "No se puede crear el artículo. Ya existe un artículo con MOCACO '{$request->mocaco}' y Campaña '{$request->campana}' (ID: {$existing->id}, Familia: {$existing->familia}).");
        }

        $article = Article::create($request->all());

        // Log the action
        AuditLog::logAction(
            'create_article',
            $article,
            Auth::user(),
            [],
            $article->toArray(),
            "Nuevo artículo creado"
        );

        return redirect()->route('articles.index')
            ->with('success', 'Artículo creado exitosamente.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Article $article)
    {
        $this->authorizeAction('view');

        return view('articles.show', compact('article'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Article $article)
    {
        $this->authorizeAction('update');

        return view('articles.edit', compact('article'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Article $article)
    {
        $this->authorizeAction('update');

        $request->validate([
            'mocaco' => 'required|string|max:20',
            'seccion' => 'nullable|string|max:50',
            'familia' => 'nullable|string|max:100',
            'cadena' => 'nullable|string|max:50',
            'mercado_origen_articulo' => 'nullable|string|max:100',
            'precio_pvp_maximo_temporada' => 'nullable|numeric',
            'partida_arancelaria' => 'nullable|string|max:50',
            'composition' => 'nullable|string',
            'campana' => 'nullable|string|max:50',
            'peso_unitario' => 'nullable|numeric',
            'grupo_arancelario' => 'nullable|string|max:50',
        ]);

        // Verificar si ya existe otro artículo con la misma combinación MOCACO + CAMPANA (excepto el actual)
        $existing = Article::where('mocaco', $request->mocaco)
            ->where('campana', $request->campana)
            ->where('id', '!=', $article->id)
            ->first();

        if ($existing) {
            return back()
                ->withInput()
                ->with('error', "No se puede actualizar el artículo. Ya existe otro artículo con MOCACO '{$request->mocaco}' y Campaña '{$request->campana}' (ID: {$existing->id}).");
        }

        $oldData = $article->toArray();
        $article->update($request->all());

        // Log the action
        AuditLog::logAction(
            'update_article',
            $article,
            Auth::user(),
            $oldData,
            $article->toArray(),
            "Artículo actualizado"
        );

        return redirect()->route('articles.index')
            ->with('success', 'Artículo actualizado exitosamente.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Article $article)
    {
        $this->authorizeAction('delete');

        $articleData = $article->toArray();
        $article->delete();

        // Log the action
        AuditLog::logAction(
            'delete_article',
            $article,
            Auth::user(),
            $articleData,
            [],
            "Artículo eliminado"
        );

        return redirect()->route('articles.index')
            ->with('success', 'Artículo eliminado exitosamente.');
    }

    /**
     * Import articles from CSV (Excel exported as CSV UTF-8) with headers:
     * id,mocaco,seccion,familia,cadena,mercado_origen_articulo,precio_pvp_maximo_temporada,partida_arancelaria,composition,campana,peso_unitario,grupo_arancelario
     * Upsert by (mocaco,campana).
     */
    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:csv,txt|max:10240',
        ]);

        $file = $request->file('file');
        $path = $file->getRealPath();

        $handle = fopen($path, 'r');
        if ($handle === false) {
            return back()->with('error', 'No se pudo abrir el archivo.');
        }

        $header = fgetcsv($handle);
        if (!$header) {
            fclose($handle);
            return back()->with('error', 'El archivo CSV está vacío.');
        }

        // Normalizar encabezados y remover BOM si existe
        $normalizedHeader = array_map(function ($h) {
            // Remover BOM UTF-8 si existe
            $h = str_replace("\xEF\xBB\xBF", '', $h);
            return strtolower(trim($h));
        }, $header);

        $required = ['mocaco', 'seccion', 'familia', 'cadena', 'mercado_origen_articulo', 'precio_pvp_maximo_temporada', 'partida_arancelaria', 'composition', 'campana', 'peso_unitario', 'grupo_arancelario'];

        // Debug: verificar qué columnas se encontraron
        $missingColumns = [];
        foreach ($required as $col) {
            if (!in_array($col, $normalizedHeader, true)) {
                $missingColumns[] = $col;
            }
        }

        if (!empty($missingColumns)) {
            fclose($handle);
            $errorMsg = "Faltan las siguientes columnas requeridas: " . implode(', ', $missingColumns) . "\n\n";
            $errorMsg .= "Columnas encontradas en el archivo: " . implode(', ', $normalizedHeader);
            return back()->with('error', $errorMsg);
        }

        $rows = [];
        $line = 1;
        $skippedLines = [];

        while (($data = fgetcsv($handle)) !== false) {
            $line++;

            // Skip completely empty rows
            if (empty(array_filter($data, function ($value) {
                return trim($value) !== '';
            }))) {
                continue;
            }

            // Validate that data has same number of columns as header
            if (count($data) !== count($normalizedHeader)) {
                $skippedLines[] = "Línea {$line}: tiene " . count($data) . " columnas, esperaba " . count($normalizedHeader);
                Log::warning("CSV: línea {$line} tiene " . count($data) . " columnas, pero el encabezado tiene " . count($normalizedHeader) . " columnas. Saltando esta línea.");
                continue;
            }

            $row = array_combine($normalizedHeader, $data);
            if ($row === false) {
                $skippedLines[] = "Línea {$line}: error al combinar datos";
                Log::warning("CSV: no se pudo combinar columnas en línea {$line}");
                continue;
            }

            // Skip rows with empty mocaco
            if (empty(trim($row['mocaco'] ?? ''))) {
                continue;
            }

            $rows[] = [
                'mocaco' => trim($row['mocaco'] ?? ''),
                'seccion' => $row['seccion'] ?? null,
                'familia' => $row['familia'] ?? null,
                'cadena' => $row['cadena'] ?? null,
                'mercado_origen_articulo' => $row['mercado_origen_articulo'] ?? null,
                'precio_pvp_maximo_temporada' => is_numeric($row['precio_pvp_maximo_temporada'] ?? null) ? (float)$row['precio_pvp_maximo_temporada'] : null,
                'partida_arancelaria' => $row['partida_arancelaria'] ?? null,
                'composition' => $row['composition'] ?? null,
                'campana' => $row['campana'] ?? null,
                'peso_unitario' => is_numeric($row['peso_unitario'] ?? null) ? (float)$row['peso_unitario'] : null,
                'grupo_arancelario' => $row['grupo_arancelario'] ?? null,
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }
        fclose($handle);

        if (empty($rows)) {
            $errorMsg = "No se encontraron filas válidas en el CSV.\n\n";
            $errorMsg .= "Total de líneas procesadas: " . ($line - 1) . "\n";

            if (!empty($skippedLines)) {
                $errorMsg .= "Líneas saltadas:\n" . implode("\n", array_slice($skippedLines, 0, 10));
                if (count($skippedLines) > 10) {
                    $errorMsg .= "\n... y " . (count($skippedLines) - 10) . " más.";
                }
            } else {
                $errorMsg .= "\nVerifique que:\n";
                $errorMsg .= "- El archivo tenga datos después del encabezado\n";
                $errorMsg .= "- Las columnas coincidan con el encabezado\n";
                $errorMsg .= "- El campo 'mocaco' no esté vacío";
            }

            return back()->with('error', $errorMsg);
        }

        // Validar duplicados en el mismo archivo antes de insertar
        $seen = [];
        $duplicates = [];
        foreach ($rows as $idx => $row) {
            $key = $row['mocaco'] . '|' . ($row['campana'] ?? '');
            if (isset($seen[$key])) {
                $duplicates[] = "Línea " . ($idx + 2) . ": MOCACO '{$row['mocaco']}' + Campaña '{$row['campana']}' ya existe en línea " . ($seen[$key] + 2);
            } else {
                $seen[$key] = $idx;
            }
        }

        if (!empty($duplicates)) {
            $duplicatesList = implode('\n░ ', array_slice($duplicates, 0, 10));
            if (count($duplicates) > 10) {
                $duplicatesList .= '\n... y ' . (count($duplicates) - 10) . ' más.';
            }
            return back()->with('error', 'Se encontraron duplicados dentro del mismo archivo:\n' . $duplicatesList . '\n\nPor favor, corrija el archivo antes de importar.');
        }

        // Procesar filas: NO actualizar si existe, solo crear si NO existe
        try {
            DB::beginTransaction();

            $imported = 0;
            $rejected = [];
            $lineNumber = 2; // Empieza en 2 porque la línea 1 es el encabezado

            foreach ($rows as $row) {
                // Verificar si ya existe en la base de datos
                $existing = Article::where('mocaco', $row['mocaco'])
                    ->where('campana', $row['campana'])
                    ->first();

                if ($existing) {
                    // RECHAZAR: ya existe, NO se actualiza
                    $rejected[] = [
                        'line' => $lineNumber,
                        'mocaco' => $row['mocaco'],
                        'campana' => $row['campana'] ?? '(vacío)',
                        'familia' => $row['familia'] ?? '',
                        'composition' => $row['composition'] ?? '',
                        'existing_id' => $existing->id,
                        'existing_familia' => $existing->familia ?? '',
                        'existing_composition' => $existing->composition ?? '',
                        'existing_created_at' => $existing->created_at->format('d/m/Y H:i')
                    ];
                } else {
                    // CREAR: no existe, se puede crear
                    try {
                        Article::create($row);
                        $imported++;
                    } catch (\Exception $e) {
                        // Si falla por índice único (por si acaso)
                        $rejected[] = [
                            'line' => $lineNumber,
                            'mocaco' => $row['mocaco'],
                            'campana' => $row['campana'] ?? '(vacío)',
                            'familia' => $row['familia'] ?? '',
                            'composition' => $row['composition'] ?? '',
                            'error' => $e->getMessage()
                        ];
                        Log::warning("Error al crear artículo en línea {$lineNumber}: " . $e->getMessage());
                    }
                }
                $lineNumber++;
            }

            DB::commit();

            // Preparar mensaje de resultado
            $message = "Importación completada.\n";
            $message .= "✓ {$imported} artículos importados correctamente.\n";

            if (!empty($rejected)) {
                $message .= "✗ " . count($rejected) . " artículos rechazados por duplicados (MOCACO + CAMPANA ya existe en la base de datos).";
            }

            if (!empty($rejected)) {
                return back()
                    ->with('success', $message)
                    ->with('imported_count', $imported)
                    ->with('rejected_count', count($rejected))
                    ->with('rejected_items', $rejected);
            }

            return back()->with('success', $message);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al importar artículos: ' . $e->getMessage());
            return back()->with('error', 'Error al importar: ' . $e->getMessage());
        }
    }

    /**
     * Descargar CSV de ejemplo con encabezados correctos.
     */
    public function importSample()
    {
        $headers = [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => 'attachment; filename="articles_sample.csv"',
        ];

        $columns = [
            'mocaco',
            'seccion',
            'familia',
            'cadena',
            'mercado_origen_articulo',
            'precio_pvp_maximo_temporada',
            'partida_arancelaria',
            'composition',
            'campana',
            'peso_unitario',
            'grupo_arancelario',
        ];

        $examples = [
            [
                '3555313809',
                'TEXTIL',
                'T-SHIRT',
                'ZARA',
                'UE',
                '19.99',
                '6109100000',
                '100% COTON',
                'I2023',
                '0.250',
                '6204',
            ],
            [
                '3555313810',
                'TEXTIL',
                'PANTALON',
                'ZARA',
                'UE',
                '29.99',
                '6203420000',
                '98% COTON 2% ELASTANE',
                'I2023',
                '0.350',
                '6203',
            ],
            [
                '3555313811',
                'TEXTIL',
                'VESTIDO',
                'ZARA',
                'UE',
                '39.99',
                '6204430000',
                '100% POLYESTER',
                'I2023',
                '0.280',
                '6204',
            ],
            [
                '3555313812',
                'CALZADO',
                'ZAPATOS',
                'ZARA',
                'UE',
                '49.99',
                '6403990000',
                'CUERO',
                'I2023',
                '0.450',
                '6403',
            ],
            [
                '3555313813',
                'TEXTIL',
                'CAMISA',
                'ZARA',
                'UE',
                '24.99',
                '6205200000',
                '100% COTON',
                'I2023',
                '0.220',
                '6205',
            ],
        ];

        $callback = function () use ($columns, $examples) {
            $output = fopen('php://output', 'w');
            // BOM para Excel
            fwrite($output, "\xEF\xBB\xBF");
            fputcsv($output, $columns);
            foreach ($examples as $example) {
                fputcsv($output, $example);
            }
            fclose($output);
        };

        return response()->stream($callback, 200, $headers);
    }
}
