<?php

namespace App\Http\Controllers;

use App\Models\Brand;
use Illuminate\Http\Request;
use App\Traits\AuthorizesResourceAccess;

class BrandController extends Controller
{
    use AuthorizesResourceAccess;

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $this->authorizeAction('view');
        $brands = Brand::orderBy('name')->paginate(20);
        return view('brands.index', compact('brands'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $this->authorizeAction('create');
        return view('brands.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->authorizeAction('create');
        $validated = $request->validate([
            'name' => 'required|string|unique:brands,name|max:255',
            'aliases' => 'nullable|string', // Comma separated
            'description' => 'nullable|string',
        ]);

        // Process aliases from string to array
        $aliases = [];
        if (!empty($validated['aliases'])) {
            $aliases = array_map('trim', explode(',', $validated['aliases']));
            $aliases = array_filter($aliases); // Remove empty strings
        }

        Brand::create([
            'name' => $validated['name'],
            'aliases' => $aliases,
            'description' => $validated['description'],
            'is_active' => true,
        ]);

        return redirect()->route('brands.index')->with('success', 'Marca creada correctamente.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Brand $brand)
    {
        $this->authorizeAction('update');
        return view('brands.edit', compact('brand'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Brand $brand)
    {
        $this->authorizeAction('update');
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:brands,name,' . $brand->id,
            'aliases' => 'nullable|string',
            'description' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        // Process aliases
        $aliases = [];
        if (!empty($validated['aliases'])) {
            $aliases = array_map('trim', explode(',', $validated['aliases']));
            $aliases = array_filter($aliases);
        }

        $brand->update([
            'name' => $validated['name'],
            'aliases' => $aliases,
            'description' => $validated['description'],
            'is_active' => $request->has('is_active'),
        ]);

        return redirect()->route('brands.index')->with('success', 'Marca actualizada correctamente.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Brand $brand)
    {
        $this->authorizeAction('delete');
        $brand->delete();
        return redirect()->route('brands.index')->with('success', 'Marca eliminada correctamente.');
    }
}
