<?php

namespace App\Http\Controllers;

use App\Models\Client;
use App\Models\AuditLog;
use App\Traits\AuthorizesResourceAccess;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ClientController extends Controller
{
    use AuthorizesResourceAccess;
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $this->authorizeAction('view');
        
        $query = Client::query();

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('client_name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%");
            });
        }

        $perPage = $request->get('per_page', 20);
        $clients = $query->orderBy('client_name')->paginate($perPage)->appends($request->except('page'));

        return view('clients.index', compact('clients'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $this->authorizeAction('create');
        
        return view('clients.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->authorizeAction('create');
        
        $request->validate([
            'client_name' => 'required|string|max:100',
            'email' => 'nullable|email|max:100',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
        ]);

        $client = Client::create(array_merge($request->all(), ['is_active' => true]));

        // Log the action
        AuditLog::logAction(
            'create_client',
            $client,
            Auth::user(),
            [],
            $client->toArray(),
            "Nuevo cliente creado"
        );

        return redirect()->route('clients.index')
            ->with('success', 'Cliente creado exitosamente.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Client $client)
    {
        $this->authorizeAction('view');
        
        $client->load('orders');
        return view('clients.show', compact('client'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Client $client)
    {
        $this->authorizeAction('update');
        
        return view('clients.edit', compact('client'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Client $client)
    {
        $this->authorizeAction('update');
        
        $request->validate([
            'client_name' => 'required|string|max:100',
            'email' => 'nullable|email|max:100',
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
        ]);

        $oldData = $client->toArray();
        $client->update(array_merge($request->all(), ['is_active' => true]));

        // Log the action
        AuditLog::logAction(
            'update_client',
            $client,
            Auth::user(),
            $oldData,
            $client->toArray(),
            "Cliente actualizado"
        );

        return redirect()->route('clients.index')
            ->with('success', 'Cliente actualizado exitosamente.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Client $client)
    {
        $this->authorizeAction('delete');
        
        $clientData = $client->toArray();
        $client->delete();

        // Log the action
        AuditLog::logAction(
            'delete_client',
            $client,
            Auth::user(),
            $clientData,
            [],
            "Cliente eliminado"
        );

        return redirect()->route('clients.index')
            ->with('success', 'Cliente eliminado exitosamente.');
    }
}
