<?php

namespace App\Http\Controllers;

use App\Models\ContainerEntry;
use App\Models\TheoreticalArticle;
use App\Models\User;
use App\Models\ImportProgress;
use App\Services\ContainerEntryService;
use App\Imports\TheoreticalArticleImport;
use App\Jobs\ProcessTheoreticalImportJob;
use App\Enums\ContainerEntryStatus;
use App\Traits\AuthorizesResourceAccess;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;
use App\Exports\ContainerEntriesUnprocessedExport;
use App\Exports\TheoreticalArticleSampleExport;

class ContainerEntryController extends Controller
{
    use AuthorizesResourceAccess;
    protected $service;

    public function __construct(ContainerEntryService $service)
    {
        $this->service = $service;
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $this->authorizeAction('view');

        $query = ContainerEntry::with(['usuarioAsignado', 'usuariosAsignados', 'theoreticalArticles']);

        // Si no es admin, solo mostrar asignados o disponibles
        if (!Auth::user()->hasRole('admin')) {
            $query->where(function ($q) {
                $q->where('usuario_asignado_id', Auth::id())
                    ->orWhereNull('usuario_asignado_id')
                    ->orWhereHas('usuariosAsignados', function ($qq) {
                        $qq->where('user_id', Auth::id());
                    });
            });
        }

        // Filtros
        if ($request->filled('estado')) {
            $query->where('estado', $request->estado);
        }

        if ($request->filled('n_camion')) {
            $query->where('n_camion', 'like', '%' . $request->n_camion . '%');
        }

        if ($request->filled('usuario_asignado_id')) {
            $userId = $request->usuario_asignado_id;
            $query->where(function ($q) use ($userId) {
                $q->where('usuario_asignado_id', $userId)
                    ->orWhereHas('usuariosAsignados', function ($qq) use ($userId) {
                        $qq->where('user_id', $userId);
                    });
            });
        }

        $perPage = $request->get('per_page', 20);
        $containerEntries = $query->orderBy('created_at', 'desc')->paginate($perPage)->appends($request->except('page'));

        // Calcular progreso para cada entrada
        foreach ($containerEntries as $entry) {
            $entry->progress = $this->service->calculateProgress($entry);
            if ($entry->fecha_inicio_proceso && $entry->fecha_finalizacion) {
                $entry->cycle_time = $this->service->calculateCycleTime($entry);
            }
        }

        $users = User::all();

        return view('container-entries.index', compact('containerEntries', 'users'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $this->authorizeAction('create');

        $users = User::all();
        return view('container-entries.create', compact('users'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->authorizeAction('create');

        $validated = $request->validate([
            'n_camion' => 'required|string|max:50|unique:container_entries,n_camion',
            'marca' => 'nullable|string|max:100',
            'tipo_producto' => 'nullable|string|max:100',
            'usuario_asignado_id' => 'nullable|exists:users,id',
            'notes' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            $containerEntry = ContainerEntry::create([
                ...$validated,
                'estado' => ContainerEntryStatus::EN_PROCESO,
                'fecha_inicio_proceso' => now(),
            ]);

            DB::commit();

            return redirect()->route('container-entries.show', $containerEntry)
                ->with('success', 'Entrada de contenedor creada correctamente.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al crear entrada de contenedor: ' . $e->getMessage());

            return back()->withInput()
                ->with('error', 'Error al crear la entrada: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(ContainerEntry $containerEntry)
    {
        $this->authorizeAction('view');

        $containerEntry->load(['usuarioAsignado', 'theoreticalArticles']);

        $progress = $this->service->calculateProgress($containerEntry);
        $cycleTime = $this->service->calculateCycleTime($containerEntry);
        $latestImport = \App\Models\ImportProgress::where('container_entry_id', $containerEntry->id)->latest()->first();

        return view('container-entries.show', compact('containerEntry', 'progress', 'cycleTime', 'latestImport'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ContainerEntry $containerEntry)
    {
        $this->authorizeAction('update');

        $users = User::all();
        return view('container-entries.edit', compact('containerEntry', 'users'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ContainerEntry $containerEntry)
    {
        $this->authorizeAction('update');

        $validated = $request->validate([
            'n_camion' => 'required|string|max:50|unique:container_entries,n_camion,' . $containerEntry->id,
            'marca' => 'nullable|string|max:100',
            'tipo_producto' => 'nullable|string|max:100',
            'usuario_asignado_id' => 'nullable|exists:users,id',
            'notes' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            // Si se está asignando un usuario, verificar que no esté en otros contenedores
            if (isset($validated['usuario_asignado_id']) && $validated['usuario_asignado_id'] !== null) {
                $userId = $validated['usuario_asignado_id'];
                $user = \App\Models\User::find($userId);
                $containersWithUser = [];

                // Verificar en la tabla pivote
                $otherContainers = ContainerEntry::enProceso()
                    ->where('id', '!=', $containerEntry->id)
                    ->whereHas('usuariosAsignados', function ($query) use ($userId) {
                        $query->where('users.id', $userId);
                    })
                    ->get();

                foreach ($otherContainers as $otherContainer) {
                    $containersWithUser[] = $otherContainer;
                }

                // Verificar en el campo usuario_asignado_id
                $containersWithUserId = ContainerEntry::enProceso()
                    ->where('id', '!=', $containerEntry->id)
                    ->where('usuario_asignado_id', $userId)
                    ->get();

                foreach ($containersWithUserId as $otherContainer) {
                    $exists = collect($containersWithUser)->contains(function ($c) use ($otherContainer) {
                        return $c->id === $otherContainer->id;
                    });
                    if (!$exists) {
                        $containersWithUser[] = $otherContainer;
                    }
                }

                // Si el usuario está en otros contenedores, NO asignarlo
                if (!empty($containersWithUser)) {
                    $containerNames = collect($containersWithUser)
                        ->map(fn($c) => "Contenedor #{$c->n_camion}")
                        ->implode(', ');
                    // No actualizar el usuario_asignado_id, mantener el que ya estaba
                    $validated['usuario_asignado_id'] = $containerEntry->usuario_asignado_id;
                    session()->flash('warning', "El usuario {$user->name} ya está asignado en: {$containerNames}. No se asignó a este contenedor.");
                } else {
                    // Si no está en otros contenedores, agregar a la tabla pivote para consistencia
                    if (!$containerEntry->usuariosAsignados()->where('users.id', $userId)->exists()) {
                        $containerEntry->usuariosAsignados()->attach($userId);
                    }
                }
            }

            $containerEntry->update($validated);

            DB::commit();

            return redirect()->route('container-entries.show', $containerEntry)
                ->with('success', 'Entrada actualizada correctamente.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al actualizar entrada: ' . $e->getMessage());

            return back()->withInput()
                ->with('error', 'Error al actualizar: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ContainerEntry $containerEntry)
    {
        $this->authorizeAction('delete');

        try {
            $containerEntry->delete();

            return redirect()->route('container-entries.index')
                ->with('success', 'Entrada eliminada correctamente.');
        } catch (\Exception $e) {
            Log::error('Error al eliminar entrada: ' . $e->getMessage());

            return back()->with('error', 'Error al eliminar: ' . $e->getMessage());
        }
    }

    /**
     * Asignar usuario a un contenedor (método legacy, usar assignUsers para múltiples)
     * Un usuario solo puede estar asignado a un contenedor a la vez.
     * Si el usuario ya está asignado a otro contenedor, se desasigna del anterior y se asigna al nuevo.
     */
    public function assignUser(Request $request, ContainerEntry $containerEntry)
    {
        $this->authorizeAction('assign');

        $validated = $request->validate([
            'usuario_asignado_id' => 'required|exists:users,id',
        ]);

        try {
            DB::beginTransaction();

            $userId = $validated['usuario_asignado_id'];
            $user = \App\Models\User::find($userId);
            $containersWithUser = [];

            // Verificar si el usuario está asignado a otros contenedores (cualquier estado, incluyendo finalizados)
            // 1. Buscar en la tabla pivote
            $otherContainers = ContainerEntry::where('id', '!=', $containerEntry->id)
                ->whereHas('usuariosAsignados', function ($query) use ($userId) {
                    $query->where('users.id', $userId);
                })
                ->get();

            foreach ($otherContainers as $otherContainer) {
                $containersWithUser[] = $otherContainer;
            }

            // 2. Buscar en el campo usuario_asignado_id
            $containersWithUserId = ContainerEntry::where('id', '!=', $containerEntry->id)
                ->where('usuario_asignado_id', $userId)
                ->get();

            foreach ($containersWithUserId as $otherContainer) {
                $exists = collect($containersWithUser)->contains(function ($c) use ($otherContainer) {
                    return $c->id === $otherContainer->id;
                });
                if (!$exists) {
                    $containersWithUser[] = $otherContainer;
                }
            }

            // Si el usuario está en otros contenedores, desasignarlo de ellos
            if (!empty($containersWithUser)) {
                foreach ($containersWithUser as $otherContainer) {
                    // Desasignar de la tabla pivote
                    $otherContainer->usuariosAsignados()->detach($userId);

                    // Si era el usuario_asignado_id, limpiarlo
                    if ($otherContainer->usuario_asignado_id == $userId) {
                        $otherContainer->update(['usuario_asignado_id' => null]);
                    }
                }

                $containerNames = collect($containersWithUser)
                    ->map(fn($c) => "Contenedor #{$c->n_camion}")
                    ->implode(', ');
            }

            // Asignar al nuevo contenedor
            $containerEntry->update([
                'usuario_asignado_id' => $userId,
            ]);

            // También agregar a la tabla pivote para consistencia
            if (!$containerEntry->usuariosAsignados()->where('users.id', $userId)->exists()) {
                $containerEntry->usuariosAsignados()->attach($userId);
            }

            DB::commit();

            if (!empty($containersWithUser)) {
                $message = "Usuario {$user->name} movido desde {$containerNames} y asignado correctamente al contenedor #{$containerEntry->n_camion}.";
                return back()->with('success', $message);
            } else {
                $message = "Usuario {$user->name} asignado correctamente al contenedor #{$containerEntry->n_camion}.";
                return back()->with('success', $message);
            }
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al asignar usuario: ' . $e->getMessage());

            return back()->with('error', 'Error al asignar usuario: ' . $e->getMessage());
        }
    }

    /**
     * Asignar múltiples usuarios a un contenedor (sincroniza el pivote)
     * Un usuario solo puede estar asignado a un contenedor a la vez.
     * Si un usuario ya está asignado a otro contenedor, se desasigna del anterior y se asigna al nuevo.
     */
    public function assignUsers(Request $request, ContainerEntry $containerEntry)
    {
        $this->authorizeAction('assign');

        $validated = $request->validate([
            'usuario_ids' => 'required|array',
            'usuario_ids.*' => 'exists:users,id',
        ]);

        try {
            DB::beginTransaction();

            $ids = collect($validated['usuario_ids'])->unique()->values()->all();
            $messages = [];
            $users = \App\Models\User::whereIn('id', $ids)->get()->keyBy('id');
            $movedUsers = [];
            $newUsers = [];

            // Procesar cada usuario
            foreach ($ids as $userId) {
                $user = $users[$userId];
                $containersWithUser = [];

                // Buscar otros contenedores donde este usuario esté asignado (cualquier estado, incluyendo finalizados)
                // 1. Buscar en la tabla pivote
                $otherContainers = ContainerEntry::where('id', '!=', $containerEntry->id)
                    ->whereHas('usuariosAsignados', function ($query) use ($userId) {
                        $query->where('users.id', $userId);
                    })
                    ->get();

                foreach ($otherContainers as $otherContainer) {
                    $containersWithUser[] = $otherContainer;
                }

                // 2. Buscar en el campo usuario_asignado_id
                $containersWithUserId = ContainerEntry::where('id', '!=', $containerEntry->id)
                    ->where('usuario_asignado_id', $userId)
                    ->get();

                foreach ($containersWithUserId as $otherContainer) {
                    $exists = collect($containersWithUser)->contains(function ($c) use ($otherContainer) {
                        return $c->id === $otherContainer->id;
                    });
                    if (!$exists) {
                        $containersWithUser[] = $otherContainer;
                    }
                }

                // Si el usuario está en otros contenedores, desasignarlo de ellos
                if (!empty($containersWithUser)) {
                    foreach ($containersWithUser as $otherContainer) {
                        // Desasignar de la tabla pivote
                        $otherContainer->usuariosAsignados()->detach($userId);

                        // Si era el usuario_asignado_id, limpiarlo
                        if ($otherContainer->usuario_asignado_id == $userId) {
                            $otherContainer->update(['usuario_asignado_id' => null]);
                        }
                    }

                    $containerNames = collect($containersWithUser)
                        ->map(fn($c) => "Contenedor #{$c->n_camion}")
                        ->implode(', ');

                    $movedUsers[] = $userId;
                    $messages[] = "Usuario {$user->name} movido desde {$containerNames}.";
                } else {
                    $newUsers[] = $userId;
                }
            }

            // Asignar todos los usuarios al nuevo contenedor
            // Obtener usuarios actualmente asignados al contenedor
            $currentUserIds = $containerEntry->usuariosAsignados()->pluck('users.id')->toArray();

            // Combinar usuarios actuales con los nuevos (sin duplicados)
            $allUserIds = array_unique(array_merge($currentUserIds, $ids));

            $containerEntry->usuariosAsignados()->sync($allUserIds);

            DB::commit();

            // Construir mensaje
            $totalAssigned = count($ids);
            $movedCount = count($movedUsers);
            $newCount = count($newUsers);

            if ($movedCount > 0 && $newCount > 0) {
                $message = "{$totalAssigned} usuario(s) asignado(s) correctamente al contenedor #{$containerEntry->n_camion}. " . implode(' ', $messages);
                return back()->with('success', $message);
            } elseif ($movedCount > 0) {
                $message = "{$totalAssigned} usuario(s) movido(s) y asignado(s) correctamente al contenedor #{$containerEntry->n_camion}. " . implode(' ', $messages);
                return back()->with('success', $message);
            } else {
                $message = "{$totalAssigned} usuario(s) asignado(s) correctamente al contenedor #{$containerEntry->n_camion}.";
                return back()->with('success', $message);
            }
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al asignar múltiples usuarios: ' . $e->getMessage());
            return back()->with('error', 'Error al asignar múltiples usuarios: ' . $e->getMessage());
        }
    }

    /**
     * Cargar Excel de artículos teóricos (Async con Queue)
     */
    public function uploadTheoreticalArticles(Request $request, ContainerEntry $containerEntry)
    {
        $this->authorizeAction('upload');

        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls,csv|max:102400', // Increased to 100MB
        ]);

        try {
            DB::beginTransaction();

            $file = $request->file('file');
            $filename = $file->getClientOriginalName();

            // Store file temporarily
            $filePath = $file->store('imports/theoretical', 'local');

            // Create import progress record
            $importProgress = ImportProgress::create([
                'container_entry_id' => $containerEntry->id,
                'filename' => $filename,
                'status' => 'pending',
                'total_rows' => 0,
                'processed_rows' => 0,
            ]);

            // Dispatch job to queue
            ProcessTheoreticalImportJob::dispatch($importProgress->id, $filePath);

            DB::commit();

            return back()->with(
                'success',
                "Archivo '{$filename}' cargado correctamente. La importación se está procesando en segundo plano. Puedes ver el progreso en esta página."
            )->with('import_progress_id', $importProgress->id);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al cargar archivo para importación: ' . $e->getMessage());

            return back()->with('error', 'Error al cargar archivo: ' . $e->getMessage());
        }
    }

    /**
     * Get import progress status (AJAX endpoint)
     */
    public function importProgress(Request $request, ContainerEntry $containerEntry)
    {
        $this->authorizeAction('view');

        $importProgressId = $request->get('import_progress_id');

        if (!$importProgressId) {
            // Get latest import for this container
            $importProgress = ImportProgress::where('container_entry_id', $containerEntry->id)
                ->latest()
                ->first();
        } else {
            $importProgress = ImportProgress::find($importProgressId);
        }

        if (!$importProgress) {
            return response()->json([
                'status' => 'not_found',
                'message' => 'No se encontró el progreso de importación',
            ], 404);
        }

        return response()->json([
            'id' => $importProgress->id,
            'status' => $importProgress->status,
            'filename' => $importProgress->filename,
            'total_rows' => $importProgress->total_rows,
            'processed_rows' => $importProgress->processed_rows,
            'imported_rows' => $importProgress->imported_rows,
            'skipped_rows' => $importProgress->skipped_rows,
            'failed_rows' => $importProgress->failed_rows,
            'progress_percentage' => $importProgress->progress_percentage,
            'started_at' => $importProgress->started_at?->format('Y-m-d H:i:s'),
            'completed_at' => $importProgress->completed_at?->format('Y-m-d H:i:s'),
            'error_message' => $importProgress->error_message,
            'is_complete' => $importProgress->isComplete(),
            'is_failed' => $importProgress->isFailed(),
            'is_processing' => $importProgress->isProcessing(),
        ]);
    }

    /**
     * Cargar Inventario Masivo (Excel)
     * Carga artículos directamente al inventario (inventory table)
     */
    public function uploadInventory(Request $request, ContainerEntry $containerEntry)
    {
        $this->authorizeAction('upload');

        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls,csv|max:10240',
        ]);

        try {
            DB::beginTransaction();

            $import = new \App\Imports\InventoryImport($containerEntry);
            Excel::import($import, $request->file('file'));

            DB::commit();

            $stats = $import->getStats();

            return back()->with(
                'success',
                "Inventario cargado correctamente. {$stats['imported']} items creados, {$stats['skipped']} omitidos."
            );
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al cargar inventario masivo: ' . $e->getMessage());

            return back()->with('error', 'Error al cargar inventario: ' . $e->getMessage());
        }
    }

    /**
     * Finalizar contenedor con reporte de discrepancias
     */
    public function finalize(Request $request, ContainerEntry $containerEntry)
    {
        $this->authorizeAction('finalize');

        $force = $request->boolean('force', false);

        // Siempre generar el reporte primero para mostrarlo
        $report = $this->service->generateDiscrepancyReport($containerEntry);

        $result = $this->service->finalizeContainer($containerEntry, $force);

        if ($result['success']) {
            return redirect()->route('container-entries.show', $containerEntry)
                ->with('success', $result['message'])
                ->with('discrepancy_report', $result['report'] ?? $report);
        } else {
            // Siempre mostrar el reporte incluso cuando hay error
            return redirect()->route('container-entries.show', $containerEntry)
                ->with('error', $result['message'])
                ->with('discrepancy_report', $result['report'] ?? $report);
        }
    }

    /**
     * Descargar reporte de discrepancias en Excel
     */
    public function discrepancyReport(ContainerEntry $containerEntry)
    {
        $this->authorizeAction('view');

        $filename = 'reporte_discrepancias_' . $containerEntry->n_camion . '_' . date('Y-m-d_His') . '.xlsx';
        return Excel::download(new \App\Exports\DiscrepancyReportExport($containerEntry), $filename);
    }

    /**
     * Obtener progreso del contenedor
     */
    public function progress(ContainerEntry $containerEntry)
    {
        $this->authorizeAction('view');

        $progress = $this->service->calculateProgress($containerEntry);

        return response()->json($progress);
    }

    /**
     * Exportar contenedores no procesados (estado: En Proceso) a XLSX
     */
    public function exportUnprocessed(Request $request)
    {
        $this->authorizeAction('export');

        $filename = 'contenedores_no_procesados_' . date('Y-m-d_His') . '.xlsx';
        return Excel::download(new ContainerEntriesUnprocessedExport(), $filename);
    }

    /**
     * Descargar plantilla XLSX para artículos teóricos
     */
    public function downloadTheoreticalTemplate()
    {
        $this->authorizeAction('view');

        $filename = 'plantilla_articulos_teoricos.xlsx';
        return Excel::download(new TheoreticalArticleSampleExport(), $filename);
    }

    /**
     * Descargar plantilla XLSX para carga masiva de inventario
     */
    public function downloadInventoryTemplate()
    {
        $this->authorizeAction('view');

        $filename = 'plantilla_inventario_masivo.xlsx';
        return Excel::download(new \App\Exports\InventorySampleExport(), $filename);
    }
}
