<?php

namespace App\Http\Controllers;

use App\Models\Inventory;
use App\Models\Article;
use App\Models\AuditLog;
use App\Exports\InventoryExport;
use App\Exports\InventoryGroupedExport;
use App\Traits\AuthorizesResourceAccess;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;

class InventoryController extends Controller
{
    use AuthorizesResourceAccess;
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $this->authorizeAction('view');

        $query = Inventory::with('article');

        // Apply filters
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('full_barcode', 'like', "%{$search}%")
                    ->orWhere('mocaco', 'like', "%{$search}%")
                    ->orWhere('n_carton', 'like', "%{$search}%")
                    ->orWhere('ubicacion', 'like', "%{$search}%");
            });
        }

        if ($request->filled('box_number')) {
            $query->where('n_carton', 'like', "%{$request->box_number}%");
        }

        if ($request->filled('location')) {
            $query->where('ubicacion', 'like', "%{$request->location}%");
        }

        $inventory = $query->orderBy('created_at', 'desc')->paginate(20);

        return view('inventory.index', compact('inventory'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $this->authorizeAction('create');

        $articles = Article::orderBy('mocaco')->get();
        return view('inventory.create', compact('articles'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->authorizeAction('create');

        $request->validate([
            'full_barcode' => 'required|string|max:255',
            'mocaco' => 'required|string|max:50',
            'n_carton' => 'required|string|max:50',
            'season_int' => 'nullable|string|max:50',
            'notes' => 'nullable|string',
            'conteneur' => 'nullable|string|max:100',
            'n_id_operario' => 'nullable|string|max:50',
            'ubicacion' => 'nullable|string|max:100',
            'categoria_seleccionada' => 'nullable|string|max:100',
            'famillie_usuario' => 'nullable|string|max:100',
            'seccion' => 'nullable|string|max:100',
            'familia_articulo_description' => 'nullable|string|max:255',
            'cadena' => 'nullable|string|max:100',
            'mercado_origen_articulo' => 'nullable|string|max:100',
            'precio_pvp_maximo_temporada' => 'nullable|numeric',
            'partida_arancelaria' => 'nullable|string|max:100',
            'composition' => 'nullable|string|max:255',
            'campana' => 'nullable|string|max:100',
            'peso_unitario' => 'nullable|numeric',
            'grupo_arancelario' => 'nullable|string|max:100',
            'detail_usuario' => 'nullable|string|max:255',
        ]);

        $inventory = Inventory::create($request->all());

        // Log the action
        AuditLog::logAction(
            'create_inventory',
            $inventory,
            Auth::user(),
            [],
            $inventory->toArray(),
            "Nuevo item de inventario creado"
        );

        return redirect()->route('inventory.index')
            ->with('success', 'Item de inventario creado exitosamente.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Inventory $inventory)
    {
        $this->authorizeAction('view');

        $inventory->load(['article', 'order.client']);
        return view('inventory.show', compact('inventory'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $this->authorizeAction('update');

        $inventory = Inventory::findOrFail($id);
        $articles = Article::orderBy('mocaco')->get();
        return view('inventory.edit', compact('inventory', 'articles'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $this->authorizeAction('update');

        // Try to find the inventory item
        $inventory = Inventory::findOrFail($id);

        $request->validate([
            'full_barcode' => 'required|string|max:255',
            'mocaco' => 'required|string|max:50',
            'n_carton' => 'required|string|max:50',
            'season_int' => 'nullable|string|max:50',
            'notes' => 'nullable|string',
            'conteneur' => 'nullable|string|max:100',
            'n_id_operario' => 'nullable|string|max:50',
            'ubicacion' => 'nullable|string|max:100',
            'categoria_seleccionada' => 'nullable|string|max:100',
            'famillie_usuario' => 'nullable|string|max:100',
            'seccion' => 'nullable|string|max:100',
            'familia_articulo_description' => 'nullable|string|max:255',
            'cadena' => 'nullable|string|max:100',
            'mercado_origen_articulo' => 'nullable|string|max:100',
            'precio_pvp_maximo_temporada' => 'nullable|numeric',
            'partida_arancelaria' => 'nullable|string|max:100',
            'composition' => 'nullable|string|max:255',
            'campana' => 'nullable|string|max:100',
            'peso_unitario' => 'nullable|numeric',
            'grupo_arancelario' => 'nullable|string|max:100',
            'detail_usuario' => 'nullable|string|max:255',
        ]);

        $oldData = $inventory->toArray();
        $inventory->update($request->all());

        // Log the action
        AuditLog::logAction(
            'update_inventory',
            $inventory,
            Auth::user(),
            $oldData,
            $inventory->toArray(),
            "Item de inventario actualizado"
        );

        return redirect()->route('inventory.index')
            ->with('success', 'Item de inventario actualizado exitosamente.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Inventory $inventory)
    {
        $this->authorizeAction('delete');

        $inventoryData = $inventory->toArray();
        $inventory->delete();

        // Log the action
        AuditLog::logAction(
            'delete_inventory',
            $inventory,
            Auth::user(),
            $inventoryData,
            [],
            "Item de inventario eliminado"
        );

        return redirect()->route('inventory.index')
            ->with('success', 'Item de inventario eliminado exitosamente.');
    }

    /**
     * Search inventory with filters and pagination (API endpoint).
     */
    public function search(Request $request): JsonResponse
    {
        $request->validate([
            'barcode' => 'nullable|string|max:50',
            'mocaco' => 'nullable|string|max:20',
            'n_carton' => 'nullable|string|max:50',
            'ubicacion' => 'nullable|string|max:50',
            'marcas' => 'nullable|string|max:50',
            'containers' => 'nullable|array',
            'containers.*' => 'string',
            'seasons' => 'nullable|array',
            'seasons.*' => 'string',
            'needs_review' => 'nullable|boolean',
            'page' => 'nullable|integer|min:1',
            'limit' => 'nullable|integer|min:1|max:250',
        ]);

        $query = Inventory::with(['article', 'order.client']);
        // Ocultar ítems despachados del inventario activo
        $query->where(function ($q) {
            $q->whereNull('status')
                ->orWhereNotIn('status', ['dispatched', 'despachado']);
        });

        // Apply filters
        if ($request->barcode) {
            $query->byBarcode($request->barcode);
        }

        if ($request->mocaco) {
            $query->where('mocaco', 'like', "%{$request->mocaco}%");
        }

        if ($request->n_carton) {
            $query->where('n_carton', 'like', "%{$request->n_carton}%");
        }

        if ($request->ubicacion) {
            $query->where('ubicacion', 'like', "%{$request->ubicacion}%");
        }

        if ($request->marcas) {
            $query->where('cadena', 'like', "%{$request->marcas}%");
        }

        // Multiselect filter for containers
        if ($request->filled('containers') && is_array($request->containers) && count($request->containers) > 0) {
            $query->whereIn('conteneur', $request->containers);
        }

        // Multiselect filter for seasons
        if ($request->filled('seasons') && is_array($request->seasons) && count($request->seasons) > 0) {
            $query->whereIn('season_int', $request->seasons);
        }

        // Filtro para artículos "Sin Info" (needs_review)
        if ($request->filled('needs_review') && $request->needs_review) {
            $query->where('needs_review', 1);
        }

        // Get total count
        $totalRecords = $query->count();

        // Apply pagination
        $page = $request->input('page', 1);
        $limit = $request->input('limit', 50);
        $offset = ($page - 1) * $limit;

        // Default ordering by fecha_escaneo DESC (most recent first)
        $results = $query->orderBy('fecha_escaneo', 'desc')
            ->orderBy('n_carton')
            ->orderBy('full_barcode')
            ->orderBy('mocaco')
            ->orderBy('detail_usuario')
            ->offset($offset)
            ->limit($limit)
            ->get();

        // Format status display and add cliente and pedido info for each result
        $results = $results->map(function ($item) {
            $statusRaw = strtolower($item->status ?? 'disponible');
            $statusDisplay = ucfirst($statusRaw);
            $clientName = '';
            $orderIdDisplay = '';

            // Obtener información del pedido y cliente si existe order_id
            if ($item->order_id) {
                $orderIdDisplay = $item->order_id;

                try {
                    // Primero intentar cargar la relación si no está cargada
                    if (!$item->relationLoaded('order')) {
                        $item->load('order.client');
                    }

                    // Si tenemos la relación order cargada
                    if ($item->order) {
                        // Asegurar que el cliente esté cargado
                        if (!$item->order->relationLoaded('client')) {
                            $item->order->load('client');
                        }

                        // Obtener el nombre del cliente
                        if ($item->order->client) {
                            $clientName = $item->order->client->client_name ?? '';
                        }
                    } else {
                        // Si la relación no se cargó, intentar cargar directamente desde la BD
                        $order = \App\Models\Order::with('client')->find($item->order_id);
                        if ($order && $order->client) {
                            $clientName = $order->client->client_name ?? '';
                        }
                    }
                } catch (\Exception $e) {
                    // Si hay error, mantener vacío
                }

                // Si está reservado, actualizar el status display
                if ($statusRaw === 'reservado') {
                    $displayClient = $clientName ?: 'N/A';
                    $statusDisplay = "Reservado & {$displayClient} & Pedido #{$orderIdDisplay}";
                }
            }

            $itemArray = $item->toArray();
            $itemArray['status_display'] = $statusDisplay;
            $itemArray['cliente'] = $clientName;
            $itemArray['pedido_id'] = $orderIdDisplay;
            return $itemArray;
        });

        return response()->json([
            'success' => true,
            'results' => $results,
            'totalRecords' => $totalRecords,
            'page' => $page,
            'limit' => $limit,
        ]);
    }

    /**
     * Update inventory item field (API endpoint).
     */
    public function updateField(Request $request): JsonResponse
    {
        $this->authorizeAction('update');

        $request->validate([
            'id' => 'required|exists:inventory,id',
            'column' => 'required|string',
            'value' => 'nullable|string',
        ]);

        try {
            $inventory = Inventory::findOrFail($request->id);
            $oldValue = $inventory->getAttribute($request->column);

            $inventory->update([
                $request->column => $request->value
            ]);

            // Log the action
            AuditLog::logAction(
                'update_inventory',
                $inventory,
                Auth::user(),
                [$request->column => $oldValue],
                [$request->column => $request->value],
                "Campo {$request->column} actualizado"
            );

            return response()->json([
                'success' => true,
                'message' => 'Artículo actualizado correctamente.',
                'inventory' => $inventory,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al actualizar el artículo: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Delete inventory item.
     */
    public function delete(Request $request): JsonResponse
    {
        $this->authorizeAction('delete');

        $request->validate([
            'id' => 'required|exists:inventory,id',
        ]);

        try {
            $inventory = Inventory::findOrFail($request->id);
            $inventoryData = $inventory->toArray();

            $inventory->delete();

            // Log the action
            AuditLog::logAction(
                'delete_inventory',
                $inventory,
                Auth::user(),
                $inventoryData,
                [],
                "Artículo eliminado del inventario"
            );

            return response()->json([
                'success' => true,
                'message' => 'Artículo eliminado correctamente.',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar el artículo: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get inventory statistics.
     */
    public function getStatistics(): JsonResponse
    {
        $stats = [
            'total_items' => Inventory::count(),
            'total_boxes' => Inventory::distinct('n_carton')->count(),
            'total_containers' => Inventory::distinct('conteneur')->count(),
            'items_by_category' => Inventory::selectRaw('categoria_seleccionada, COUNT(*) as count')
                ->groupBy('categoria_seleccionada')
                ->get(),
            'items_by_family' => Inventory::selectRaw('famillie_usuario, COUNT(*) as count')
                ->groupBy('famillie_usuario')
                ->orderBy('count', 'desc')
                ->limit(10)
                ->get(),
            'recent_scans' => Inventory::orderBy('fecha_escaneo', 'desc')
                ->limit(10)
                ->get(['full_barcode', 'mocaco', 'n_carton', 'fecha_escaneo']),
        ];

        return response()->json([
            'success' => true,
            'statistics' => $stats,
        ]);
    }

    /**
     * Get unique filter options for multiselect dropdowns.
     */
    public function getFilterOptions(): JsonResponse
    {
        $containers = Inventory::select('conteneur')
            ->whereNotNull('conteneur')
            ->where('conteneur', '!=', '')
            ->distinct()
            ->orderBy('conteneur')
            ->pluck('conteneur')
            ->values();

        $seasons = Inventory::select('season_int')
            ->whereNotNull('season_int')
            ->where('season_int', '!=', '')
            ->distinct()
            ->orderBy('season_int')
            ->pluck('season_int')
            ->values();

        $brands = Inventory::select('cadena')
            ->whereNotNull('cadena')
            ->where('cadena', '!=', '')
            ->distinct()
            ->orderBy('cadena')
            ->pluck('cadena')
            ->values();

        return response()->json([
            'success' => true,
            'containers' => $containers,
            'seasons' => $seasons,
            'brands' => $brands,
        ]);
    }

    /**
     * Export inventory to Excel format.
     */
    /**
     * Exportar inventario usando FastExcel para mejor rendimiento con grandes volúmenes.
     */
    public function export(Request $request)
    {
        $this->authorizeAction('view');

        $filename = 'inventario_export_' . date('Y-m-d_His') . '.xlsx';

        // Generator function to yield items one by one
        $inventoryGenerator = function () use ($request) {
            $query = Inventory::query();

            // Ocultar ítems despachados en la exportación
            $query->where(function ($q) {
                $q->whereNull('status')
                    ->orWhereNotIn('status', ['dispatched', 'despachado']);
            });

            // Apply same filters as index
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('full_barcode', 'like', "%{$search}%")
                        ->orWhere('mocaco', 'like', "%{$search}%")
                        ->orWhere('n_carton', 'like', "%{$search}%")
                        ->orWhere('ubicacion', 'like', "%{$search}%");
                });
            }

            if ($request->filled('box_number')) {
                $query->where('n_carton', 'like', "%{$request->box_number}%");
            }

            if ($request->filled('location')) {
                $query->where('ubicacion', 'like', "%{$request->location}%");
            }

            // Multiselect filter for containers
            if ($request->filled('containers') && is_array($request->containers) && count($request->containers) > 0) {
                $query->whereIn('conteneur', $request->containers);
            }

            // Multiselect filter for seasons
            if ($request->filled('seasons') && is_array($request->seasons) && count($request->seasons) > 0) {
                $query->whereIn('season_int', $request->seasons);
            }

            // Brand filter
            if ($request->filled('marcas')) {
                $query->where('cadena', 'like', "%{$request->marcas}%");
            }

            // Needs review filter
            if ($request->filled('needs_review') && $request->needs_review) {
                $query->where('needs_review', 1);
            }

            $query->orderBy('id', 'asc');

            foreach ($query->cursor() as $item) {
                yield [
                    'ID' => $item->id,
                    'Código de Barras' => $item->full_barcode,
                    'MOCACO' => $item->mocaco,
                    'Nº Carton' => $item->n_carton,
                    'Ubicación' => $item->ubicacion ?? '',
                    'Contenedor' => $item->conteneur ?? '',
                    'Operario' => $item->n_id_operario ?? '',
                    'Season INT' => $item->season_int ?? '',
                    'Categoría' => $item->categoria_seleccionada ?? '',
                    'Familia Usuario' => $item->famillie_usuario ?? '',
                    'Detail Usuario' => $item->detail_usuario ?? '',
                    'Sección' => $item->seccion ?? '',
                    'Familia Artículo' => $item->familia_articulo_description ?? '',
                    'Cadena' => $item->cadena ?? '',
                    'Mercado Origen' => $item->mercado_origen_articulo ?? '',
                    'Precio PVP Max' => $item->precio_pvp_maximo_temporada ?? '',
                    'Partida Arancelaria' => $item->partida_arancelaria ?? '',
                    'Composition' => $item->composition ?? '',
                    'Campaña' => $item->campana ?? '',
                    'Peso Unitario' => $item->peso_unitario ?? '',
                    'Grupo Arancelario' => $item->grupo_arancelario ?? '',
                    'Talla' => $item->size ?? '',
                    'Color' => $item->color ?? '',
                    'Status' => $item->status ?? 'disponible',
                    'Order ID' => $item->order_id ?? '',
                    'Notas' => $item->notes ?? '',
                    'Fecha Escaneo' => $item->fecha_escaneo ? $item->fecha_escaneo->format('d/m/Y H:i:s') : '',
                    'Fecha Creación' => $item->created_at ? $item->created_at->format('d/m/Y H:i:s') : '',
                    'Última Actualización' => $item->updated_at ? $item->updated_at->format('d/m/Y H:i:s') : '',
                ];
            }
        };

        return (new \Rap2hpoutre\FastExcel\FastExcel($inventoryGenerator()))->download($filename);
    }

    /**
     * Exportar inventario agrupado (XLSX)
     * Agrupa por campos del artículo y suma unidades
     */
    public function exportGrouped(Request $request)
    {
        $this->authorizeAction('view');

        return Excel::download(new InventoryGroupedExport($request->all()), 'inventario_agrupado_' . now()->format('Y-m-d_His') . '.xlsx');
    }

    /**
     * Buscar cajas sin ubicación asignada
     */
    public function boxesWithoutLocation(Request $request)
    {
        $this->authorizeAction('view');

        // Obtener cajas únicas sin ubicación
        $query = Inventory::select(
            'n_carton',
            'conteneur',
            DB::raw('COUNT(*) as total_items'),
            DB::raw('MIN(fecha_escaneo) as first_scan'),
            DB::raw('MAX(fecha_escaneo) as last_scan'),
            DB::raw('GROUP_CONCAT(DISTINCT n_id_operario) as operators')
        )
            ->where(function ($q) {
                $q->whereNull('ubicacion')
                    ->orWhere('ubicacion', '')
                    ->orWhere('ubicacion', 'N/A');
            })
            ->whereIn('status', ['disponible', 'available'])
            ->groupBy('n_carton', 'conteneur')
            ->orderBy('last_scan', 'desc');

        // Filtros
        if ($request->filled('conteneur')) {
            $query->where('conteneur', 'like', '%' . $request->conteneur . '%');
        }

        if ($request->filled('n_carton')) {
            $query->where('n_carton', 'like', '%' . $request->n_carton . '%');
        }

        if ($request->filled('date_from')) {
            $query->havingRaw('last_scan >= ?', [$request->date_from . ' 00:00:00']);
        }

        if ($request->filled('date_to')) {
            $query->havingRaw('last_scan <= ?', [$request->date_to . ' 23:59:59']);
        }

        $boxes = $query->paginate(50);

        // Calcular estadísticas
        $stats = [
            'total_boxes' => Inventory::select('n_carton')
                ->where(function ($q) {
                    $q->whereNull('ubicacion')
                        ->orWhere('ubicacion', '')
                        ->orWhere('ubicacion', 'N/A');
                })
                ->whereIn('status', ['disponible', 'available'])
                ->distinct()
                ->count('n_carton'),
            'total_items' => Inventory::where(function ($q) {
                $q->whereNull('ubicacion')
                    ->orWhere('ubicacion', '')
                    ->orWhere('ubicacion', 'N/A');
            })
                ->whereIn('status', ['disponible', 'available'])
                ->count(),
        ];

        return view('inventory.boxes-without-location', compact('boxes', 'stats'));
    }

    /**
     * Exportar inventario filtrado por marca/cadena
     */
    public function exportByBrand(Request $request)
    {
        $this->authorizeAction('view');

        $brand = $request->input('brand');

        $filename = 'inventario_marca_' . ($brand ? str_replace(' ', '_', $brand) : 'todas') . '_' . now()->format('Y-m-d_His') . '.xlsx';

        return Excel::download(new \App\Exports\InventoryByBrandExport($brand), $filename);
    }
}
