<?php

namespace App\Http\Controllers;

use App\Models\Inventory;
use App\Traits\AuthorizesResourceAccess;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ProductivityReportController extends Controller
{
    use AuthorizesResourceAccess;
    /**
     * Reporte de productividad por operario
     * Agrupa por operario_id y fecha (timestamp_scan)
     */
    public function index(Request $request)
    {
        $this->authorizeAction('view');

        $query = Inventory::select(
            'n_id_operario',
            DB::raw('DATE(fecha_escaneo) as fecha'),
            DB::raw('COUNT(*) as unidades_escaneadas'),
            DB::raw('COUNT(DISTINCT n_carton) as cajas_procesadas')
        )
            ->groupBy('n_id_operario', DB::raw('DATE(fecha_escaneo)'))
            ->orderBy('fecha', 'desc')
            ->orderBy('n_id_operario');

        // Filtros
        if ($request->filled('operario_id')) {
            $query->where('n_id_operario', $request->operario_id);
        }

        if ($request->filled('fecha_desde')) {
            $query->whereDate('fecha_escaneo', '>=', $request->fecha_desde);
        }

        if ($request->filled('fecha_hasta')) {
            $query->whereDate('fecha_escaneo', '<=', $request->fecha_hasta);
        }

        $perPage = $request->get('per_page', 50);
        $reports = $query->paginate($perPage)->appends($request->except('page'));

        // Calcular totales para el resumen (sin paginación)
        $summaryQuery = Inventory::select(
            DB::raw('COUNT(*) as total_unidades'),
            DB::raw('COUNT(DISTINCT n_carton) as total_cajas')
        );

        // Aplicar mismos filtros para el resumen
        if ($request->filled('operario_id')) {
            $summaryQuery->where('n_id_operario', $request->operario_id);
        }

        if ($request->filled('fecha_desde')) {
            $summaryQuery->whereDate('fecha_escaneo', '>=', $request->fecha_desde);
        }

        if ($request->filled('fecha_hasta')) {
            $summaryQuery->whereDate('fecha_escaneo', '<=', $request->fecha_hasta);
        }

        $summary = $summaryQuery->first();

        // Obtener lista de operarios únicos para el filtro
        $operarios = Inventory::select('n_id_operario')
            ->distinct()
            ->whereNotNull('n_id_operario')
            ->orderBy('n_id_operario')
            ->pluck('n_id_operario');

        return view('reports.productivity', compact('reports', 'operarios', 'summary'));
    }

    /**
     * Exportar reporte de productividad a Excel
     */
    /**
     * Exportar reporte de productividad a Excel (XLSX)
     */
    public function export(Request $request)
    {
        $this->authorizeAction('export');

        $filename = 'reporte_productividad_' . now()->format('Y-m-d') . '.xlsx';

        return \Maatwebsite\Excel\Facades\Excel::download(new \App\Exports\ProductivityExport($request->all()), $filename);
    }
}
