<?php

namespace App\Http\Controllers;

use App\Models\BoxReservation;
use App\Models\BoxReservationItem;
use App\Models\Client;
use App\Models\Inventory;
use App\Imports\ReservationBoxesImport;
use App\Exports\InventoryAvailableExport;
use App\Exports\ReservationTemplateExport;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;

class ReservationController extends Controller
{
    /**
     * Display a listing of reservations
     */
    public function index(Request $request)
    {
        $query = BoxReservation::with(['client', 'reservedBy', 'items']);

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by client
        if ($request->filled('client_id')) {
            $query->where('client_id', $request->client_id);
        }

        $reservations = $query->orderBy('created_at', 'desc')->paginate(20);

        // Calculate stats for each reservation
        foreach ($reservations as $reservation) {
            $reservation->total_boxes = $reservation->items()->count();
            $reservation->confirmed_boxes = $reservation->confirmedItems()->count();
            $reservation->pending_boxes = $reservation->pendingItems()->count();
        }

        $clients = Client::orderBy('client_name')->get();

        return view('reservations.index', compact('reservations', 'clients'));
    }

    /**
     * Show the form for creating a new reservation
     */
    public function create()
    {
        $clients = Client::orderBy('client_name')->get();
        return view('reservations.create', compact('clients'));
    }

    /**
     * Store a newly created reservation
     */
    public function store(Request $request)
    {
        $request->validate([
            'client_id' => 'required|exists:clients,id',
            'notes' => 'nullable|string',
        ]);

        try {
            DB::beginTransaction();

            $reservation = BoxReservation::create([
                'client_id' => $request->client_id,
                'reservation_code' => BoxReservation::generateReservationCode(),
                'status' => 'pending',
                'reserved_by_user_id' => auth()->id(),
                'notes' => $request->notes,
                'reserved_at' => now(),
            ]);

            DB::commit();

            return redirect()
                ->route('reservations.show', $reservation)
                ->with('success', "Reserva {$reservation->reservation_code} creada correctamente. Ahora puedes importar las cajas.");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating reservation: ' . $e->getMessage());
            return back()->withInput()->with('error', 'Error al crear la reserva: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified reservation
     */
    public function show(BoxReservation $reservation)
    {
        $reservation->load(['client', 'reservedBy', 'confirmedBy', 'items.inventory']);

        $stats = [
            'total_boxes' => $reservation->items()->count(),
            'confirmed_boxes' => $reservation->confirmedItems()->count(),
            'pending_boxes' => $reservation->pendingItems()->count(),
            'completion_percentage' => $reservation->completion_percentage,
        ];

        return view('reservations.show', compact('reservation', 'stats'));
    }

    /**
     * Import boxes from Excel
     */
    public function importBoxes(Request $request, BoxReservation $reservation)
    {
        $request->validate([
            'file' => 'required|file|mimes:xlsx,xls,csv|max:10240',
        ]);

        // Only allow import for pending or in_progress reservations
        if ($reservation->isCompleted() || $reservation->isCancelled()) {
            return back()->with('error', 'Solo se pueden importar cajas a reservas pendientes o en proceso.');
        }

        try {
            DB::beginTransaction();

            $import = new ReservationBoxesImport($reservation->id);
            Excel::import($import, $request->file('file'));

            // Update inventory status for imported boxes
            $boxNumbers = $import->getBoxNumbers();
            if (!empty($boxNumbers)) {
                Inventory::whereIn('n_carton', $boxNumbers)
                    ->update([
                        'reservation_status' => 'reserved',
                        'status' => 'order_proposal',
                        'reserved_for_client_id' => $reservation->client_id,
                        'reservation_id' => $reservation->id,
                    ]);
            }

            DB::commit();

            $message = "Importación completada: {$import->getCount()} cajas agregadas.";
            if (!empty($import->getErrors())) {
                $message .= " Errores: " . implode(', ', array_slice($import->getErrors(), 0, 5));
            }

            return back()->with('success', $message);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error importing reservation boxes: ' . $e->getMessage());
            return back()->with('error', 'Error al importar cajas: ' . $e->getMessage());
        }
    }

    /**
     * Download template for box selection
     */
    public function downloadTemplate()
    {
        $filename = 'plantilla_reserva_' . date('Y-m-d') . '.xlsx';
        return Excel::download(new ReservationTemplateExport(), $filename);
    }

    /**
     * Download available inventory for selection
     */
    public function downloadAvailableInventory()
    {
        $filename = 'inventario_disponible_' . date('Y-m-d') . '.xlsx';
        return Excel::download(new InventoryAvailableExport(), $filename);
    }

    /**
     * Cancel a reservation
     */
    public function cancel(BoxReservation $reservation)
    {
        // Only allow cancellation for pending or in_progress reservations
        if ($reservation->isCompleted() || $reservation->isCancelled()) {
            return back()->with('error', 'No se puede cancelar una reserva completada o ya cancelada.');
        }

        try {
            DB::beginTransaction();

            // Update reservation status
            $reservation->update(['status' => 'cancelled']);

            // Get inventory IDs from items
            $inventoryIds = BoxReservationItem::where('reservation_id', $reservation->id)
                ->pluck('inventory_id')
                ->toArray();

            // Cancel all items
            BoxReservationItem::where('reservation_id', $reservation->id)
                ->update(['status' => 'cancelled']);

            // Release inventory (by Item link OR by Reservation ID link)
            Inventory::where(function ($q) use ($inventoryIds, $reservation) {
                if (!empty($inventoryIds)) {
                    $q->whereIn('id', $inventoryIds);
                }
                $q->orWhere('reservation_id', $reservation->id);
            })->update([
                'reservation_status' => 'available',
                'status' => 'disponible',
                'reserved_for_client_id' => null,
                'reservation_id' => null,
            ]);

            DB::commit();

            return redirect()
                ->route('reservations.index')
                ->with('success', "Reserva {$reservation->reservation_code} cancelada correctamente.");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error cancelling reservation: ' . $e->getMessage());
            return back()->with('error', 'Error al cancelar la reserva: ' . $e->getMessage());
        }
    }

    /**
     * Remove a specific box from reservation
     */
    public function removeBox(BoxReservation $reservation, BoxReservationItem $item)
    {
        // Only allow removal for pending or in_progress reservations
        if ($reservation->isCompleted() || $reservation->isCancelled()) {
            return back()->with('error', 'No se pueden eliminar cajas de reservas completadas o canceladas.');
        }

        try {
            DB::beginTransaction();

            $boxNumber = $item->box_number;

            // Release ALL inventory items with this box number that belong to this reservation
            $releasedCount = Inventory::where('n_carton', $boxNumber)
                ->where('reservation_id', $reservation->id)
                ->update([
                    'reservation_status' => 'available',
                    'status' => 'disponible',
                    'reserved_for_client_id' => null,
                    'reservation_id' => null,
                ]);

            // Delete item
            $item->delete();

            DB::commit();

            return back()->with('success', "Caja {$boxNumber} eliminada de la reserva ({$releasedCount} unidades liberadas).");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error removing box from reservation: ' . $e->getMessage());
            return back()->with('error', 'Error al eliminar la caja: ' . $e->getMessage());
        }
    }
    /**
     * Add a single box to reservation
     */
    public function addBox(Request $request, BoxReservation $reservation)
    {
        $request->validate([
            'box_number' => 'required|string',
        ]);

        if ($reservation->isCompleted() || $reservation->isCancelled()) {
            return back()->with('error', 'No se pueden agregar cajas a reservas completadas o canceladas.');
        }

        try {
            DB::beginTransaction();

            $boxNumber = trim($request->box_number);

            // Get ALL inventory items with this box number that are available
            $inventoryItems = Inventory::where('n_carton', $boxNumber)
                ->where('reservation_status', 'available')
                ->get();

            if ($inventoryItems->isEmpty()) {
                // Check if it exists but is not available
                $exists = Inventory::where('n_carton', $boxNumber)->exists();
                if ($exists) {
                    return back()->with('error', "La caja {$boxNumber} no está disponible (ya reservada o despachada).");
                }
                return back()->with('error', "La caja {$boxNumber} no existe en el sistema.");
            }

            $totalUnits = $inventoryItems->count();

            // Create ONE BoxReservationItem representing all units of this box
            BoxReservationItem::create([
                'reservation_id' => $reservation->id,
                'inventory_id' => $inventoryItems->first()->id, // Reference the first item
                'box_number' => $boxNumber,
                'quantity' => $totalUnits, // Total quantity of units
                'status' => 'confirmed',
                'scanned_at' => now(),
                'notes' => "Agregada manualmente ({$totalUnits} unidades)",
            ]);

            // Update ALL inventory items with this box number
            Inventory::where('n_carton', $boxNumber)
                ->where('reservation_status', 'available')
                ->update([
                    'reservation_status' => 'reserved',
                    'status' => 'order_proposal',
                    'reserved_for_client_id' => $reservation->client_id,
                    'reservation_id' => $reservation->id,
                ]);

            DB::commit();

            return back()->with('success', "Caja {$boxNumber} agregada correctamente ({$totalUnits} unidades).");
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error adding box to reservation: ' . $e->getMessage());
            return back()->with('error', 'Error al agregar la caja: ' . $e->getMessage());
        }
    }

    /**
     * Update reservation details
     */
    public function update(Request $request, BoxReservation $reservation)
    {
        $request->validate([
            'client_id' => 'required|exists:clients,id',
            'notes' => 'nullable|string',
        ]);

        if ($reservation->isCompleted() || $reservation->isCancelled()) {
            return back()->with('error', 'No se puede editar una reserva completada o cancelada.');
        }

        $reservation->update([
            'client_id' => $request->client_id,
            'notes' => $request->notes,
        ]);

        return back()->with('success', 'Detalles de la reserva actualizados.');
    }
}
