<?php

namespace App\Http\Controllers;

use App\Models\Article;
use App\Models\Inventory;
use App\Models\AuditLog;
use App\Models\ContainerEntry;
use App\Models\ScannerAlert;
use App\Models\ScannerOption;
use App\Services\BoxLabelService;
use Dompdf\Dompdf;
use Dompdf\Options;
use App\Services\BoxNumberService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ScannerController extends Controller
{
    /**
     * Display the scanner interface.
     */
    public function index()
    {
        // Obtener el contenedor asignado al usuario autenticado
        $assignedContainer = null;
        $user = Auth::user();

        if ($user) {
            // Usar la relación del modelo User para obtener el contenedor asignado
            $assignedContainer = $user->assignedContainer();
        }

        // Fetch dynamic options with DISTINCT to avoid duplicates
        $seasons = ScannerOption::where('type', 'season')
            ->where('active', true)
            ->select('value', 'label')
            ->distinct()
            ->orderBy('label')
            ->get();

        $categories = ScannerOption::where('type', 'category')
            ->where('active', true)
            ->select('value', 'label')
            ->distinct()
            ->orderBy('label')
            ->get();

        $families = ScannerOption::where('type', 'family')
            ->where('active', true)
            ->select('value', 'label')
            ->distinct()
            ->orderBy('label')
            ->get();

        $details = ScannerOption::where('type', 'detail')
            ->where('active', true)
            ->select('value', 'label')
            ->distinct()
            ->orderBy('label')
            ->get();

        $brands = ScannerOption::where('type', 'brand')
            ->where('active', true)
            ->select('value', 'label')
            ->distinct()
            ->orderBy('label')
            ->get();

        $sections = ScannerOption::where('type', 'section')
            ->where('active', true)
            ->select('value', 'label')
            ->distinct()
            ->orderBy('label')
            ->get();

        return view('scanner.index', [
            'assignedContainer' => $assignedContainer,
            'seasons' => $seasons,
            'categories' => $categories,
            'families' => $families,
            'details' => $details,
            'brands' => $brands,
            'sections' => $sections,
        ]);
    }

    /**
     * Search for article by barcode.
     */
    public function searchArticle(Request $request): JsonResponse
    {
        try {
            $request->validate([
                'full_barcode' => 'required|string|min:13|max:14|regex:/^\d+$/',
            ]);

            $fullBarcode = $request->input('full_barcode');
            $mocaco = Article::extractMocacoFromBarcode($fullBarcode);

            // 1) Buscar primero en la tabla legacy exactamente como en procesar_scan.php
            try {
                $legacyRows = DB::table('articulos')
                    ->where('MOCACO', $mocaco)
                    ->get();

                if ($legacyRows->isNotEmpty()) {
                    $normalized = $legacyRows->map(function ($row) {
                        // Intentar encontrar el artículo en la tabla nueva para obtener el ID
                        // Buscar usando múltiples campos para mayor precisión
                        $query = Article::where('mocaco', $row->MOCACO ?? null);

                        if ($row->SECCION ?? null) {
                            $query->where('seccion', $row->SECCION);
                        }
                        if ($row->FAMILIA ?? null) {
                            $query->where('familia', $row->FAMILIA);
                        }
                        if ($row->CAMPANA ?? null) {
                            $query->where('campana', $row->CAMPANA);
                        }

                        $article = $query->first();

                        return [
                            'id' => $article ? $article->id : null,
                            'mocaco' => $row->MOCACO ?? null,
                            'seccion' => $row->SECCION ?? null,
                            'familia' => $row->FAMILIA ?? null,
                            'cadena' => $row->CADENA ?? null,
                            'mercado_origen_articulo' => $row->MERCADO_ORIGEN_ARTICULO ?? null,
                            'precio_pvp_maximo_temporada' => $row->PRECIO_PVP_MAXIMO_TEMPORADA ?? null,
                            'partida_arancelaria' => $row->PARTIDA_ARANCELARIA ?? null,
                            'composition' => $row->COMPOSITION ?? null,
                            'campana' => $row->CAMPANA ?? null,
                            'peso_unitario' => $row->PESO_UNITARIO ?? null,
                            'grupo_arancelario' => $row->GRUPO_ARANCELARIO ?? null,
                        ];
                    });

                    return response()->json([
                        'success' => true,
                        'articles' => $normalized,
                        'message' => 'Búsqueda completada (legacy).',
                        'source' => 'legacy',
                    ]);
                }
            } catch (\Throwable $e) {
                Log::warning('Legacy search failed: ' . $e->getMessage());
            }

            // 2) Si no hay en legacy, buscar en la tabla nueva
            // Buscar por MOCACO (puede haber múltiples resultados si hay diferentes campañas)
            $articles = Article::where('mocaco', $mocaco)->get();

            // 3) Fallback: Si no hay en Article, buscar en Inventory (para ítems creados manualmente/Sin Info)
            if ($articles->isEmpty()) {
                $fallbackItem = $this->findInInventoryFallback($fullBarcode);
                if ($fallbackItem) {
                    return response()->json([
                        'success' => true,
                        'articles' => [$fallbackItem],
                        'message' => 'Artículo encontrado en inventario (previamente registrado manual).',
                        'source' => 'inventory_fallback'
                    ]);
                }
            }

            // Asegurar que los artículos incluyan el ID al serializar
            // Capturar fullBarcode para usar en el map
            $fullBarcode = $request->input('full_barcode');

            // Normalizar los resultados
            $articlesArray = $articles->map(function ($article) use ($fullBarcode) {
                return [
                    'id' => $article->id,
                    'mocaco' => $article->mocaco,
                    'seccion' => $article->seccion,
                    'familia' => $article->familia,
                    'cadena' => $article->cadena,
                    'mercado_origen_articulo' => $article->mercado_origen_articulo,
                    'precio_pvp_maximo_temporada' => $article->precio_pvp_maximo_temporada,
                    'partida_arancelaria' => $article->partida_arancelaria,
                    'composition' => $article->composition,
                    'campana' => $article->campana,
                    'peso_unitario' => $article->peso_unitario,
                    'grupo_arancelario' => $article->grupo_arancelario,
                    // Extraer talla y color del código de barras
                    'extracted_color' => Article::extractColorFromBarcode($fullBarcode),
                    'extracted_size' => Article::extractSizeFromBarcode($fullBarcode),
                ];
            });

            return response()->json([
                'success' => true,
                'articles' => $articlesArray,
                'message' => 'Búsqueda completada.',
            ]);
        } catch (\Exception $e) {
            Log::error('Error en searchArticle: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error al buscar artículo: ' . $e->getMessage(),
                'articles' => []
            ], 500);
        }
    }

    /**
     * Helper to search valid inventory item when article is not found
     */
    private function findInInventoryFallback($fullBarcode)
    {
        // Buscar en inventario el registro más reciente con ese código de barras
        // que no sea una entrada "vacía" o provisional fallida (tiene que tener mocaco, etc.)
        $inventoryItem = Inventory::where('full_barcode', $fullBarcode)
            ->whereNotNull('mocaco')
            ->where('mocaco', '!=', '')
            ->orderBy('created_at', 'desc')
            ->first();

        if ($inventoryItem) {
            return [
                'id' => null, // No tiene ID de Article real
                'mocaco' => $inventoryItem->mocaco,
                'seccion' => $inventoryItem->seccion,
                'familia' => $inventoryItem->familia_articulo_description,
                'cadena' => $inventoryItem->cadena,
                'mercado_origen_articulo' => $inventoryItem->mercado_origen_articulo,
                'precio_pvp_maximo_temporada' => $inventoryItem->precio_pvp_maximo_temporada,
                'partida_arancelaria' => $inventoryItem->partida_arancelaria,
                'composition' => $inventoryItem->composition,
                'campana' => $inventoryItem->campana,
                'peso_unitario' => $inventoryItem->peso_unitario,
                'grupo_arancelario' => $inventoryItem->grupo_arancelario,
                'extracted_color' => Article::extractColorFromBarcode($fullBarcode),
                'extracted_size' => Article::extractSizeFromBarcode($fullBarcode),
                'is_fallback' => true // Flag para identificar origen
            ];
        }

        return null;
    }

    /**
     * Process scanned barcode and save to inventory.
     */
    public function processScan(Request $request): JsonResponse
    {
        $request->validate([
            'full_barcode' => 'required|string|min:13|max:14|regex:/^\d+$/',
            'article_id' => 'nullable|integer|exists:articles,id',
            'mocaco' => 'required|string|max:20',
            'n_carton' => 'required|string|max:50',
            'season_int' => 'required|string|max:20',
            'conteneur' => 'required|string|max:50',
            'categoria_seleccionada' => 'required|string|max:20',
            'famillie_usuario' => 'required|string|max:50',
            'detail_usuario' => 'nullable|string|max:100',
            'ubicacion' => 'nullable|string|max:50',
            'notes' => 'nullable|string',
            'needs_review' => 'nullable|boolean',
            'marca_manual' => 'nullable|string|max:20',
            'seccion_manual' => 'nullable|string|max:50',
        ]);

        try {
            DB::beginTransaction();

            // Get article data - buscar por ID si está disponible, sino por MOCACO
            Log::info('processScan recibido', [
                'article_id' => $request->article_id,
                'mocaco' => $request->mocaco,
                'needs_review' => $request->needs_review
            ]);

            $article = null;
            if ($request->has('article_id') && $request->article_id) {
                $article = Article::find($request->article_id);
            } else {
                $article = Article::where('mocaco', $request->mocaco)->first();
            }

            if (!$article) {
                // Feature: Crear artículo automáticamente si se confirma desde "Sin Info"
                if ($request->create_article && $request->marca_manual && $request->seccion_manual) {
                    $newMocaco = $request->mocaco;
                    // Si el mocaco viene vacío o es genérico, intentar usar la marca manual o extraerlo
                    if (empty($newMocaco)) {
                        $newMocaco = Article::extractMocacoFromBarcode($request->full_barcode) ?? $request->marca_manual;
                    }

                    try {
                        $article = Article::create([
                            'mocaco' => $newMocaco,
                            'seccion' => $request->seccion_manual,
                            'familia' => 'SIN INFO', // Default
                            'cadena' => $request->marca_manual,
                            'mercado_origen_articulo' => 'MANUAL',
                            'precio_pvp_maximo_temporada' => 0,
                            'composition' => 'MANUAL',
                        ]);

                        Log::info('Artículo creado desde Sin Info/Fallback', ['id' => $article->id]);
                    } catch (\Exception $e) {
                        Log::error('Error creando artículo desde scan:', ['error' => $e->getMessage()]);
                        // Continue to standard fallback if creation fails
                    }
                }

                if (!$article) {
                    if ($request->needs_review) {
                        Log::info('Artículo no encontrado, pero "Sin Info" activo. Creando inventario con datos manuales.');
                    } else {
                        return response()->json([
                            'success' => false,
                            'message' => 'Artículo no encontrado en la base de datos.',
                        ], 404);
                    }
                }
            } else {
                Log::info('Artículo encontrado', [
                    'id' => $article->id,
                    'mocaco' => $article->mocaco,
                ]);
            }

            // Validar que el cartón no esté despachado previamente
            $cartonDespachado = Inventory::byBox($request->n_carton)
                ->whereIn('status', ['dispatched', 'despachado'])
                ->exists();

            if ($cartonDespachado) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => 'No se puede escanear: este cartón ya está despachado.',
                ], 422);
            }

            // Verificar si existe un ContainerEntry con este contenedor y está en proceso
            $containerEntry = ContainerEntry::where('n_camion', trim($request->conteneur))
                ->where('estado', 'En Proceso')
                ->first();

            // Prepare common data
            $inventoryData = [
                'full_barcode' => trim($request->full_barcode),
                'n_carton' => trim($request->n_carton),
                'season_int' => trim($request->season_int),
                'conteneur' => trim($request->conteneur),
                'categoria_seleccionada' => trim($request->categoria_seleccionada),
                'famillie_usuario' => trim($request->famillie_usuario),
                'detail_usuario' => trim($request->detail_usuario ?? ''),
                'ubicacion' => trim($request->ubicacion ?? ''),
                'notes' => trim($request->notes ?? ''),
                'n_id_operario' => trim(Auth::user()->operario_id ?? Auth::user()->name),
                'fecha_escaneo' => now(),
                'needs_review' => $request->needs_review ?? false,
                'color' => Article::extractColorFromBarcode($request->full_barcode),
                'size' => Article::extractSizeFromBarcode($request->full_barcode),
            ];

            if ($article) {
                $inventoryData = array_merge($inventoryData, [
                    'mocaco' => trim($article->mocaco),
                    'seccion' => $article->seccion ?? '',
                    'familia_articulo_description' => $article->familia ?? '',
                    'cadena' => $article->cadena ?? '',
                    'mercado_origen_articulo' => $article->mercado_origen_articulo ?? '',
                    'precio_pvp_maximo_temporada' => $article->precio_pvp_maximo_temporada ?? 0,
                    'partida_arancelaria' => $article->partida_arancelaria ?? '',
                    'composition' => $article->composition ?? '',
                    'campana' => $article->campana ?? '',
                    'peso_unitario' => $article->peso_unitario ?? 0,
                    'grupo_arancelario' => $article->grupo_arancelario ?? '',
                ]);
            } else {
                // Fallback to manual/request data
                // MEJORA: Cuando "Sin Info" está activo, priorizar datos manuales
                $inventoryData = array_merge($inventoryData, [
                    'mocaco' => trim($request->mocaco),
                    'seccion' => trim($request->seccion_manual ?? ''),
                    'familia_articulo_description' => 'SIN INFO',
                    'cadena' => trim($request->marca_manual ?? ''),
                    'mercado_origen_articulo' => 'MANUAL',
                    'precio_pvp_maximo_temporada' => 0,
                    'partida_arancelaria' => '',
                    'composition' => 'MANUAL',
                    'campana' => '',
                    'peso_unitario' => 0,
                    'grupo_arancelario' => '',
                ]);
            }

            // Create inventory record (cada escaneo crea un registro individual)
            $inventory = Inventory::create($inventoryData);

            // Log the action
            AuditLog::logAction(
                'scan',
                $inventory,
                Auth::user(),
                [],
                $inventory->toArray(),
                "Artículo escaneado: {$request->full_barcode}" . ($request->needs_review ? " [SIN INFO]" : "")
            );

            // Verificar alertas
            $matchedAlerts = [];
            if ($article) {
                $user = Auth::user();
                $alerts = ScannerAlert::where('is_active', true)
                    ->where(function ($query) use ($user) {
                        $query->whereNull('user_id')
                            ->orWhere('user_id', $user->id);
                    })
                    ->get();

                foreach ($alerts as $alert) {
                    $textToCheck = '';

                    if ($alert->type === 'composition' && !empty($article->composition)) {
                        $textToCheck = $article->composition;
                    } elseif ($alert->type === 'country' && !empty($article->mercado_origen_articulo)) {
                        $textToCheck = $article->mercado_origen_articulo;
                    }

                    if (!empty($textToCheck) && $alert->matches($textToCheck)) {
                        $matchedAlerts[] = [
                            'id' => $alert->id,
                            'type' => $alert->type,
                            'description' => $alert->description ?? "Alerta de {$alert->type}",
                            'keywords' => $alert->keywords_array,
                        ];
                    }
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Artículo escaneado y guardado correctamente.',
                'inventory' => $inventory,
                'alerts' => $matchedAlerts,
                'has_alerts' => count($matchedAlerts) > 0,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Error al procesar el escaneo: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Eliminar el último artículo escaneado por el usuario actual
     */
    public function deleteLastScannedItem(Request $request): JsonResponse
    {
        $request->validate([
            'n_carton' => 'required|string|max:50',
            'conteneur' => 'nullable|string|max:50',
        ]);

        try {
            DB::beginTransaction();

            $user = Auth::user();
            $nCarton = trim($request->n_carton);
            $conteneur = $request->conteneur ? trim($request->conteneur) : null;

            // Buscar el último artículo escaneado por este usuario en esta caja
            $query = Inventory::byBox($nCarton)
                ->where('n_id_operario', $user->operario_id ?? $user->name);

            if ($conteneur) {
                $query->byContainer($conteneur);
            }

            // Ordenar por fecha de escaneo descendente para obtener el más reciente
            $lastItem = $query->orderBy('fecha_escaneo', 'desc')
                ->orderBy('id', 'desc')
                ->first();

            if (!$lastItem) {
                return response()->json([
                    'success' => false,
                    'message' => 'No se encontró ningún artículo escaneado por usted en esta caja.',
                ], 404);
            }

            // Verificar que el artículo no esté despachado
            if (in_array($lastItem->status, ['dispatched', 'despachado', 'reservado'])) {
                return response()->json([
                    'success' => false,
                    'message' => 'No se puede eliminar: este artículo ya está despachado o reservado.',
                ], 422);
            }

            $deletedBarcode = $lastItem->full_barcode;
            $deletedMocaco = $lastItem->mocaco;

            // Log la acción antes de eliminar
            AuditLog::logAction(
                'delete_last_scan',
                $lastItem,
                $user,
                $lastItem->toArray(),
                [],
                "Último artículo escaneado eliminado: {$deletedBarcode} (MOCACO: {$deletedMocaco})"
            );

            // Eliminar el artículo
            $lastItem->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Último artículo escaneado eliminado correctamente.',
                'deleted_item' => [
                    'full_barcode' => $deletedBarcode,
                    'mocaco' => $deletedMocaco,
                ],
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al eliminar último artículo escaneado: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar el último artículo: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Eliminar un artículo específico del contenido de una caja.
     */
    public function deleteScannedItem(Request $request): JsonResponse
    {
        $request->validate([
            'inventory_id' => 'required|integer',
            'n_carton' => 'required|string|max:50',
            'conteneur' => 'nullable|string|max:50',
        ]);

        $user = Auth::user();
        $boxNumber = trim($request->n_carton);
        $conteneur = $request->conteneur ? trim($request->conteneur) : null;

        try {
            DB::beginTransaction();

            $inventoryItem = Inventory::where('id', $request->inventory_id)
                ->lockForUpdate()
                ->first();

            if (!$inventoryItem) {
                DB::rollBack();

                return response()->json([
                    'success' => false,
                    'message' => 'El artículo seleccionado ya no existe en el inventario.',
                ], 404);
            }

            // Validar que pertenezca a la caja solicitada
            if (strcasecmp((string) $inventoryItem->n_carton, $boxNumber) !== 0) {
                DB::rollBack();

                return response()->json([
                    'success' => false,
                    'message' => 'El artículo no pertenece a la caja indicada.',
                ], 422);
            }

            if ($conteneur && strcasecmp((string) $inventoryItem->conteneur, $conteneur) !== 0) {
                DB::rollBack();

                return response()->json([
                    'success' => false,
                    'message' => 'El artículo no pertenece al contenedor indicado.',
                ], 422);
            }

            if (in_array($inventoryItem->status, ['dispatched', 'despachado', 'reservado'])) {
                DB::rollBack();

                return response()->json([
                    'success' => false,
                    'message' => 'No se puede eliminar: este artículo ya está despachado o reservado.',
                ], 422);
            }

            $operatorIdentifier = $user ? ($user->operario_id ?? $user->name) : null;
            if ($user && !$user->can('inventory.delete') && strcasecmp((string) $inventoryItem->n_id_operario, (string) $operatorIdentifier) !== 0) {
                DB::rollBack();

                return response()->json([
                    'success' => false,
                    'message' => 'No tiene permisos para eliminar este artículo.',
                ], 403);
            }

            $deletedData = [
                'full_barcode' => $inventoryItem->full_barcode,
                'mocaco' => $inventoryItem->mocaco,
                'n_carton' => $inventoryItem->n_carton,
                'conteneur' => $inventoryItem->conteneur,
                'n_id_operario' => $inventoryItem->n_id_operario,
            ];

            AuditLog::logAction(
                'delete_box_item',
                $inventoryItem,
                $user,
                $inventoryItem->toArray(),
                [],
                "Artículo eliminado desde escáner: {$inventoryItem->full_barcode} (MOCACO: {$inventoryItem->mocaco})"
            );

            $inventoryItem->delete();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Artículo eliminado correctamente de la caja.',
                'deleted_item' => $deletedData,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al eliminar artículo desde el escáner: ' . $e->getMessage(), [
                'inventory_id' => $request->inventory_id,
            ]);

            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar el artículo: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Vaciar completamente una caja (eliminar todos los artículos)
     */
    public function emptyBox(Request $request): JsonResponse
    {
        $request->validate([
            'n_carton' => 'required|string|max:50',
            'conteneur' => 'nullable|string|max:50',
        ]);

        try {
            DB::beginTransaction();

            $user = Auth::user();
            $nCarton = trim($request->n_carton);
            $conteneur = $request->conteneur ? trim($request->conteneur) : null;

            // Buscar todos los artículos de la caja
            $query = Inventory::byBox($nCarton);

            if ($conteneur) {
                $query->byContainer($conteneur);
            }

            $items = $query->get();

            if ($items->isEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => 'No se encontraron artículos en esta caja.',
                ], 404);
            }

            // Verificar que ningún artículo esté despachado o reservado
            $blockedItems = $items->filter(function ($item) {
                return in_array($item->status, ['dispatched', 'despachado', 'reservado', 'order_proposal']);
            });

            if ($blockedItems->isNotEmpty()) {
                return response()->json([
                    'success' => false,
                    'message' => "No se puede vaciar la caja: {$blockedItems->count()} artículo(s) están despachados, reservados o en propuesta de pedido.",
                ], 422);
            }

            $deletedCount = $items->count();
            $itemsSummary = $items->map(function ($item) {
                return [
                    'id' => $item->id,
                    'full_barcode' => $item->full_barcode,
                    'mocaco' => $item->mocaco,
                ];
            })->toArray();

            // Log la acción antes de eliminar
            AuditLog::logAction(
                'empty_box',
                null,
                $user,
                ['box' => $nCarton, 'container' => $conteneur, 'items_count' => $deletedCount],
                [],
                "Caja vaciada: {$nCarton} ({$deletedCount} artículos eliminados)"
            );

            // Eliminar todos los artículos
            foreach ($items as $item) {
                $item->delete();
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => "Caja {$nCarton} vaciada correctamente. {$deletedCount} artículo(s) eliminado(s).",
                'deleted_count' => $deletedCount,
                'deleted_items' => $itemsSummary,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al vaciar caja: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error al vaciar la caja: ' . $e->getMessage(),
            ], 500);
        }
    }


    /**
     * Get box content by box number.
     */
    public function getBoxContent(Request $request): JsonResponse
    {
        $request->validate([
            'n_carton' => 'required|string|max:50',
            'conteneur' => 'nullable|string|max:50',
        ]);

        // Bloquear si el cartón ya está despachado
        $cartonDespachado = Inventory::byBox($request->n_carton)
            ->whereIn('status', ['dispatched', 'despachado'])
            ->exists();

        if ($cartonDespachado) {
            return response()->json([
                'success' => false,
                'message' => 'No se puede consultar: este cartón ya está despachado.',
                'articles' => [],
                'total' => 0,
            ], 422);
        }

        // Log para debugging
        Log::info('getBoxContent - Request data:', [
            'n_carton' => $request->n_carton,
            'conteneur' => $request->conteneur,
            'n_carton_raw' => bin2hex($request->n_carton),
        ]);

        // Primero, veamos qué valores reales hay en la BD para debugging
        $sampleBoxes = Inventory::select('n_carton')
            ->distinct()
            ->limit(10)
            ->pluck('n_carton');


        Log::info('Sample n_carton values from DB:', $sampleBoxes->toArray());

        $query = Inventory::byBox($request->n_carton);

        if ($request->conteneur) {
            $query->byContainer($request->conteneur);
        }

        // Log the SQL query for debugging
        Log::info('SQL Query:', ['sql' => $query->toSql(), 'bindings' => $query->getBindings()]);

        $articles = $query->with('article')->get();

        Log::info('getBoxContent - Results count:', ['count' => $articles->count()]);

        return response()->json([
            'success' => true,
            'articles' => $articles,
            'total' => $articles->count(),
            'debug' => [
                'requested_box' => $request->n_carton,
                'requested_container' => $request->conteneur,
                'sample_boxes' => $sampleBoxes->toArray(),
            ],
        ]);
    }

    /**
     * Create new article if not found.
     */
    public function createArticle(Request $request): JsonResponse
    {
        $request->validate([
            'full_barcode' => 'required|string|max:50',
            'mocaco' => 'required|string|max:20',
            'seccion' => 'required|string|max:50',
            'familia' => 'required|string|max:100',
            'cadena' => 'required|string|max:50',
            'mercado_origen_articulo' => 'nullable|string|max:100',
            'precio_pvp_maximo_temporada' => 'nullable|numeric',
            'partida_arancelaria' => 'nullable|string|max:50',
            'composition' => 'nullable|string',
            'campana' => 'nullable|string|max:50',
            'peso_unitario' => 'nullable|numeric',
            'grupo_arancelario' => 'nullable|string|max:50',
        ]);

        try {
            DB::beginTransaction();

            $article = Article::create($request->all());

            // Log the action
            AuditLog::logAction(
                'create_article',
                $article,
                Auth::user(),
                [],
                $article->toArray(),
                "Nuevo artículo creado: {$request->mocaco}"
            );

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Artículo creado correctamente.',
                'article' => $article,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Error al crear el artículo: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get current container number.
     */
    public function getCurrentContainer()
    {
        try {
            $container = DB::table('config')
                ->where('key', 'current_container')
                ->value('value') ?? 'PT233I23';
        } catch (\Exception $e) {
            // Si la tabla no existe o hay error, usar valor por defecto
            $container = 'PT233I23';
        }

        return response($container, 200)
            ->header('Content-Type', 'text/plain');
    }

    /**
     * Generar el siguiente número de caja automáticamente
     */
    public function generateNextBoxNumber(Request $request): JsonResponse
    {
        $request->validate([
            'conteneur' => 'nullable|string|max:50',
            'format' => 'nullable|string|in:container,global',
        ]);

        try {
            $boxNumberService = new BoxNumberService();
            $conteneur = $request->input('conteneur');
            $format = $request->input('format', 'container');

            // Si no se proporciona contenedor, intentar obtenerlo del usuario asignado
            if (!$conteneur) {
                $user = Auth::user();
                if ($user) {
                    $assignedContainer = $user->assignedContainer();
                    if ($assignedContainer) {
                        $conteneur = $assignedContainer->n_camion;
                    }
                }
            }

            $nextBoxNumber = $boxNumberService->generateNextBoxNumber($conteneur, $format);
            $lastBoxNumber = $boxNumberService->getLastBoxNumberForContainer($conteneur);

            return response()->json([
                'success' => true,
                'box_number' => $nextBoxNumber,
                'last_box_number' => $lastBoxNumber,
                'conteneur' => $conteneur,
                'format' => $format,
            ]);
        } catch (\Exception $e) {
            Log::error('Error al generar número de caja: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error al generar el número de caja: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Generar etiqueta ZPL para una caja
     */
    public function generateBoxLabelZpl(Request $request): JsonResponse
    {
        $request->validate([
            'n_carton' => 'required|string|max:50',
            'conteneur' => 'nullable|string|max:50',
        ]);

        try {
            $service = new BoxLabelService();
            $zpl = $service->generateBoxLabelZpl(
                $request->n_carton,
                $request->conteneur
            );

            return response()->json([
                'success' => true,
                'zpl_content' => $zpl,
                'message' => 'Etiqueta ZPL generada correctamente.',
            ]);
        } catch (\Exception $e) {
            Log::error('Error al generar etiqueta ZPL: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error al generar la etiqueta: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Generar etiqueta PDF para una caja
     */
    public function generateBoxLabelPdf(Request $request)
    {
        $request->validate([
            'n_carton' => 'required|string|max:50',
            'conteneur' => 'nullable|string|max:50',
        ]);

        try {
            $service = new BoxLabelService();
            $html = $service->generateBoxLabelPdf(
                $request->n_carton,
                $request->conteneur
            );

            // Convertir HTML a PDF usando DomPDF
            $options = new Options();
            $options->set('isHtml5ParserEnabled', true);
            $options->set('isRemoteEnabled', true);
            $options->set('defaultFont', 'Arial');

            $dompdf = new Dompdf($options);
            $dompdf->loadHtml($html);
            $dompdf->setPaper('A4', 'portrait');
            $dompdf->render();

            $filename = 'etiqueta-caja-' . $request->n_carton . '.pdf';

            return response($dompdf->output(), 200)
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'inline; filename="' . $filename . '"');
        } catch (\Exception $e) {
            Log::error('Error al generar etiqueta PDF: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error al generar la etiqueta: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Buscar caja para reimpresión de etiqueta
     */
    public function findBoxForReprint(Request $request): JsonResponse
    {
        $request->validate([
            'n_carton' => 'required|string|max:50',
        ]);

        try {
            $service = new BoxLabelService();
            $box = $service->findBoxForReprint($request->n_carton);

            if (!$box) {
                return response()->json([
                    'success' => false,
                    'message' => 'Caja no encontrada.',
                ], 404);
            }

            return response()->json([
                'success' => true,
                'box' => $box,
            ]);
        } catch (\Exception $e) {
            Log::error('Error al buscar caja: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error al buscar la caja: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Generar PDF "CONTENU DU COLIS" para una caja
     */
    public function generateContenuDuColisPdf(Request $request)
    {
        $request->validate([
            'n_carton' => 'required|string|max:50',
            'conteneur' => 'nullable|string|max:50',
            'observations' => 'nullable|string|max:100',
        ]);

        try {
            $service = new BoxLabelService();
            $dompdf = $service->generateContenuDuColisPdf(
                $request->n_carton,
                $request->conteneur,
                $request->observations
            );

            $filename = 'contenu_du_colis_' . $request->n_carton . '_' . date('Y-m-d_His') . '.pdf';

            return response($dompdf->output(), 200)
                ->header('Content-Type', 'application/pdf')
                ->header('Content-Disposition', 'inline; filename="' . $filename . '"');
        } catch (\Exception $e) {
            Log::error('Error al generar PDF CONTENU DU COLIS: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error al generar el PDF: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Obtener alertas activas para el usuario actual (Solo Admin)
     */
    public function getAlerts(): JsonResponse
    {
        try {
            $user = Auth::user();

            // Verificar que el usuario sea admin
            if (!$user || !$user->hasRole('admin')) {
                return response()->json([
                    'success' => false,
                    'message' => 'No tiene permisos para acceder a esta funcionalidad.',
                ], 403);
            }

            // Obtener todas las alertas (admin puede ver todas)
            $alerts = ScannerAlert::orderBy('type')
                ->orderBy('description')
                ->get();

            return response()->json([
                'success' => true,
                'alerts' => $alerts,
            ]);
        } catch (\Exception $e) {
            Log::error('Error al obtener alertas: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error al obtener alertas: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Crear o actualizar una alerta (Solo Admin)
     */
    public function saveAlert(Request $request): JsonResponse
    {
        $request->validate([
            'id' => 'nullable|integer|exists:scanner_alerts,id',
            'type' => 'required|string|in:composition,country',
            'keywords' => 'required|string',
            'description' => 'nullable|string|max:255',
            'is_active' => 'boolean',
        ]);

        try {
            $user = Auth::user();

            // Verificar que el usuario sea admin
            if (!$user || !$user->hasRole('admin')) {
                return response()->json([
                    'success' => false,
                    'message' => 'No tiene permisos para realizar esta acción.',
                ], 403);
            }

            $data = [
                'type' => $request->type,
                'keywords' => trim($request->keywords),
                'description' => $request->description,
                'is_active' => $request->has('is_active') ? $request->is_active : true,
                'user_id' => null, // Alertas globales (solo admin puede crearlas)
            ];

            if ($request->has('id') && $request->id) {
                // Actualizar alerta existente
                $alert = ScannerAlert::findOrFail($request->id);
                $alert->update($data);

                return response()->json([
                    'success' => true,
                    'message' => 'Alerta actualizada correctamente.',
                    'alert' => $alert,
                ]);
            } else {
                // Crear nueva alerta
                $alert = ScannerAlert::create($data);

                return response()->json([
                    'success' => true,
                    'message' => 'Alerta creada correctamente.',
                    'alert' => $alert,
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Error al guardar alerta: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error al guardar la alerta: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Eliminar una alerta (Solo Admin)
     */
    public function deleteAlert(Request $request): JsonResponse
    {
        $request->validate([
            'id' => 'required|integer|exists:scanner_alerts,id',
        ]);

        try {
            $user = Auth::user();

            // Verificar que el usuario sea admin
            if (!$user || !$user->hasRole('admin')) {
                return response()->json([
                    'success' => false,
                    'message' => 'No tiene permisos para realizar esta acción.',
                ], 403);
            }

            $alert = ScannerAlert::findOrFail($request->id);
            $alert->delete();

            return response()->json([
                'success' => true,
                'message' => 'Alerta eliminada correctamente.',
            ]);
        } catch (\Exception $e) {
            Log::error('Error al eliminar alerta: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error al eliminar la alerta: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Verificar si un artículo coincide con alguna alerta activa
     */
    public function checkAlerts(Request $request): JsonResponse
    {
        $request->validate([
            'composition' => 'nullable|string',
            'mercado_origen_articulo' => 'nullable|string',
        ]);

        try {
            $user = Auth::user();

            // Obtener alertas activas
            $alerts = ScannerAlert::where('is_active', true)
                ->where(function ($query) use ($user) {
                    $query->whereNull('user_id')
                        ->orWhere('user_id', $user->id);
                })
                ->get();

            $matchedAlerts = [];

            foreach ($alerts as $alert) {
                $textToCheck = '';

                if ($alert->type === 'composition' && $request->has('composition')) {
                    $textToCheck = $request->composition ?? '';
                } elseif ($alert->type === 'country' && $request->has('mercado_origen_articulo')) {
                    $textToCheck = $request->mercado_origen_articulo ?? '';
                }

                if (!empty($textToCheck) && $alert->matches($textToCheck)) {
                    $matchedAlerts[] = [
                        'id' => $alert->id,
                        'type' => $alert->type,
                        'description' => $alert->description ?? "Alerta de {$alert->type}",
                        'keywords' => $alert->keywords_array,
                    ];
                }
            }

            return response()->json([
                'success' => true,
                'has_alerts' => count($matchedAlerts) > 0,
                'alerts' => $matchedAlerts,
            ]);
        } catch (\Exception $e) {
            Log::error('Error al verificar alertas: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Error al verificar alertas: ' . $e->getMessage(),
            ], 500);
        }
    }
}
