<?php

namespace App\Http\Controllers;

use App\Models\ScannerOption;
use Illuminate\Http\Request;

class ScannerOptionController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:scanner-options.view')->only(['index']);
        $this->middleware('permission:scanner-options.create')->only(['create', 'store']);
        $this->middleware('permission:scanner-options.update')->only(['edit', 'update']);
        $this->middleware('permission:scanner-options.delete')->only(['destroy']);
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = ScannerOption::query();

        if ($request->has('type') && $request->type) {
            $query->where('type', $request->type);
        }

        $options = $query->orderBy('type')->orderBy('label')->paginate(20);

        $types = ScannerOption::select('type')->distinct()->pluck('type');

        return view('scanner_options.index', compact('options', 'types'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('scanner_options.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'type' => 'required|string|in:season,category,family,detail,brand,section',
            'value' => [
                'required',
                'string',
                'max:255',
                // Validar que la combinación type+value sea única
                'unique:scanner_options,value,NULL,id,type,' . $request->type
            ],
            'label' => 'required|string|max:255',
            'active' => 'boolean',
        ]);

        $validated['active'] = $request->has('active');

        ScannerOption::create($validated);

        return redirect()->route('scanner-options.index')
            ->with('success', 'Option created successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(ScannerOption $scannerOption)
    {
        return view('scanner_options.edit', compact('scannerOption'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, ScannerOption $scannerOption)
    {
        $validated = $request->validate([
            'type' => 'required|string|in:season,category,family,detail,brand,section',
            'value' => [
                'required',
                'string',
                'max:255',
                // Validar que la combinación type+value sea única, excluyendo el registro actual
                'unique:scanner_options,value,' . $scannerOption->id . ',id,type,' . $request->type
            ],
            'label' => 'required|string|max:255',
            'active' => 'boolean',
        ]);

        $validated['active'] = $request->has('active');

        $scannerOption->update($validated);

        return redirect()->route('scanner-options.index')
            ->with('success', 'Option updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(ScannerOption $scannerOption)
    {
        $scannerOption->delete();

        return redirect()->route('scanner-options.index')
            ->with('success', 'Option deleted successfully.');
    }
}
