<?php

namespace App\Http\Controllers;

use App\Models\Inventory;
use App\Traits\AuthorizesResourceAccess;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class StockSummaryController extends Controller
{
    use AuthorizesResourceAccess;

    /**
     * Mostrar el resumen de stock por marca y temporadas.
     */
    public function index(Request $request)
    {
        $this->authorizeAction('view');

        $filters = $request->validate([
            'marca' => ['nullable', 'string', 'max:100'],
            'season_int' => ['nullable', 'string', 'max:50'],
            'season' => ['nullable', 'string', 'max:50'],
            'status_scope' => ['nullable', 'in:all,trabajada,sin_trabajar'],
        ]);

        $statusScope = $filters['status_scope'] ?? 'all';

        $baseQuery = Inventory::query()
            ->leftJoin('container_entries', 'container_entries.n_camion', '=', 'inventory.conteneur');

        if (!empty($filters['marca'])) {
            $baseQuery->where('container_entries.marca', 'like', '%' . $filters['marca'] . '%');
        }

        if (!empty($filters['season_int'])) {
            $baseQuery->where('inventory.season_int', $filters['season_int']);
        }

        if (!empty($filters['season'])) {
            $baseQuery->where('inventory.campana', $filters['season']);
        }

        if ($statusScope === 'trabajada') {
            $baseQuery->whereNotNull('inventory.status')
                ->where('inventory.status', '<>', 'disponible');
        } elseif ($statusScope === 'sin_trabajar') {
            $baseQuery->where(function ($query) {
                $query->whereNull('inventory.status')
                    ->orWhere('inventory.status', 'disponible');
            });
        }

        // Obtener todas las temporadas internas únicas para crear columnas dinámicas
        $seasonInts = (clone $baseQuery)
            ->select('inventory.season_int')
            ->whereNotNull('inventory.season_int')
            ->distinct()
            ->orderBy('inventory.season_int')
            ->pluck('season_int')
            ->filter()
            ->values();

        // Agrupar por marca y calcular totales por temporada interna
        $summaryByMarca = (clone $baseQuery)
            ->select([
                'container_entries.marca',
                'inventory.season_int',
                DB::raw('COUNT(*) as cantidad'),
                DB::raw("SUM(CASE WHEN inventory.status = 'disponible' OR inventory.status IS NULL THEN 1 ELSE 0 END) as sin_trabajar"),
            ])
            ->whereNotNull('container_entries.marca')
            ->groupBy('container_entries.marca', 'inventory.season_int')
            ->orderBy('container_entries.marca')
            ->orderBy('inventory.season_int')
            ->get();

        // Reorganizar los datos por marca
        $summary = [];
        foreach ($summaryByMarca as $row) {
            $marca = $row->marca ?? 'Sin Marca';
            if (!isset($summary[$marca])) {
                $summary[$marca] = [
                    'marca' => $marca,
                    'season_ints' => [],
                    'total_sin_trabajar' => 0,
                    'total_general' => 0,
                ];
            }
            
            $seasonInt = $row->season_int ?? 'Sin Temporada';
            $summary[$marca]['season_ints'][$seasonInt] = [
                'cantidad' => $row->cantidad,
                'sin_trabajar' => $row->sin_trabajar,
            ];
            $summary[$marca]['total_sin_trabajar'] += $row->sin_trabajar;
            $summary[$marca]['total_general'] += $row->cantidad;
        }

        // Convertir a colección para mantener compatibilidad
        $summary = collect($summary)->values();

        // Calcular totales generales
        $totals = (clone $baseQuery)
            ->select([
                DB::raw('COUNT(*) as total_mercancia'),
                DB::raw("SUM(CASE WHEN inventory.status = 'disponible' OR inventory.status IS NULL THEN 1 ELSE 0 END) as total_sin_trabajar"),
                DB::raw("SUM(CASE WHEN inventory.status IS NOT NULL AND inventory.status <> 'disponible' THEN 1 ELSE 0 END) as total_trabajada"),
            ])
            ->first();

        return view('reports.stock-summary', [
            'summary' => $summary,
            'seasonInts' => $seasonInts,
            'totals' => $totals,
            'filters' => array_merge([
                'marca' => null,
                'season_int' => null,
                'season' => null,
                'status_scope' => 'all',
            ], $filters),
        ]);
    }
}

