<?php

namespace App\Imports;

use App\Models\Article;
use App\Models\Inventory;
use App\Models\ContainerEntry;
use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithValidation;
use Maatwebsite\Excel\Concerns\WithBatchInserts;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Maatwebsite\Excel\Concerns\WithCustomValueBinder;
use PhpOffice\PhpSpreadsheet\Cell\Cell;
use PhpOffice\PhpSpreadsheet\Cell\DataType;
use PhpOffice\PhpSpreadsheet\Cell\DefaultValueBinder;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class InventoryImport extends DefaultValueBinder implements ToModel, WithHeadingRow, WithValidation, WithBatchInserts, WithChunkReading, WithCustomValueBinder
{
    /**
     * Bind value to cell - preserve leading zeros for mocaco column
     */
    public function bindValue(Cell $cell, $value)
    {
        // Si es la columna mocaco (columna A después de headers), forzar como texto
        if ($cell->getColumn() === 'A' && $cell->getRow() > 1) {
            $cell->setValueExplicit($value, DataType::TYPE_STRING);
            return true;
        }

        // Para otras columnas, usar el comportamiento por defecto
        return parent::bindValue($cell, $value);
    }
    protected $containerEntry;
    protected $importedCount = 0;
    protected $skippedCount = 0;

    public function __construct(ContainerEntry $containerEntry)
    {
        $this->containerEntry = $containerEntry;
    }

    /**
     * Normaliza un valor: convierte "sin dato" a null, maneja números grandes, etc.
     */
    protected function normalizeValue($value)
    {
        if ($value === null) {
            return null;
        }

        $str = trim((string)$value);

        // Convertir "sin dato" a null
        if (strtolower($str) === 'sin dato' || $str === '') {
            return null;
        }

        return $str;
    }

    /**
     * Convierte un número (que puede venir como número o string) a string sin perder precisión
     * IMPORTANTE: Preserva ceros a la izquierda para códigos como mocaco
     */
    protected function numberToString($value)
    {
        if ($value === null) {
            return '';
        }

        // Convertir a string primero para manejar cualquier tipo
        $str = trim((string)$value);

        // Si está vacío, retornar vacío
        if ($str === '') {
            return '';
        }

        // CRÍTICO: Si el string empieza con 0 y es numérico, es un código con ceros a la izquierda
        // NO convertir a número, retornar como string directamente
        if (preg_match('/^0\d+$/', $str)) {
            return $str;
        }

        // Si es numérico (puede venir como string numérico o como número)
        if (is_numeric($value) || is_numeric($str)) {
            // Convertir a float para manejar notación científica si existe
            $num = (float)$str;

            // Si es un entero (sin decimales), devolverlo sin decimales
            if ($num == floor($num)) {
                // Usar number_format para evitar notación científica en números grandes
                return number_format($num, 0, '.', '');
            }

            // Si tiene decimales, usar formato con decimales pero sin notación científica
            return rtrim(number_format($num, 10, '.', ''), '0');
        }

        return $str;
    }

    /**
     * Convierte un valor numérico con coma decimal a float
     */
    protected function parseDecimalValue($value)
    {
        if ($value === null || $value === '') {
            return null;
        }

        $str = trim((string)$value);

        // Convertir "sin dato" a null
        if (strtolower($str) === 'sin dato') {
            return null;
        }

        // Reemplazar coma por punto para decimales
        $str = str_replace(',', '.', $str);

        // Remover espacios
        $str = str_replace(' ', '', $str);

        if (is_numeric($str)) {
            return (float)$str;
        }

        return null;
    }

    /**
     * @param array $row
     *
     * @return \Illuminate\Database\Eloquent\Model|null
     */
    public function model(array $row)
    {
        // Normalizar los nombres de las columnas (case-insensitive)
        $normalizedRow = [];
        foreach ($row as $key => $value) {
            $normalizedRow[strtolower(trim($key))] = $value;
        }

        // Mapear campos
        $mocacoRaw = $normalizedRow['mocaco'] ?? $normalizedRow['codigo'] ?? '';
        $mocaco = trim($this->numberToString($mocacoRaw));

        if (empty($mocaco)) {
            $this->skippedCount++;
            return null;
        }

        // Obtener cantidad (si existe) - OBLIGATORIO
        $cantidadRaw = $normalizedRow['cantidad_unidades'] ?? $normalizedRow['unidades'] ?? $normalizedRow['cantidad'] ?? 1;
        $cantidad = is_numeric($cantidadRaw) ? (int)$cantidadRaw : 1;

        // Datos comunes
        $campanaRaw = $normalizedRow['campana'] ?? null;
        $campana = $this->normalizeValue($campanaRaw);

        $precioRaw = $normalizedRow['precio_pvp_maximo_temporada']
            ?? $normalizedRow['precio_pv']
            ?? $normalizedRow['precio']
            ?? null;
        $precio = $this->parseDecimalValue($precioRaw);

        $pesoRaw = $normalizedRow['peso_unitario']
            ?? $normalizedRow['peso_unit']
            ?? $normalizedRow['peso']
            ?? null;
        $peso = $this->parseDecimalValue($pesoRaw);

        $familiaRaw = $normalizedRow['familia_articulo_description']
            ?? $normalizedRow['familia_ar']
            ?? $normalizedRow['descripcion_familia']
            ?? $normalizedRow['familia']
            ?? null;

        $mercadoRaw = $normalizedRow['mercado_origen_articulo']
            ?? $normalizedRow['mercado']
            ?? $normalizedRow['mercado_origen']
            ?? null;

        $partidaRaw = $normalizedRow['partida_arancelaria']
            ?? $normalizedRow['partida_ar']
            ?? $normalizedRow['partida']
            ?? null;

        $composicionRaw = $normalizedRow['composition']
            ?? $normalizedRow['compositi']
            ?? $normalizedRow['composicion']
            ?? null;

        $grupoRaw = $normalizedRow['grupo_arancelario']
            ?? $normalizedRow['grupo_ara']
            ?? $normalizedRow['grupo']
            ?? null;

        // OBLIGATORIO
        $nCarton = $this->numberToString($normalizedRow['n_carton'] ?? $normalizedRow['caja'] ?? $normalizedRow['carton'] ?? null);
        if (empty($nCarton)) {
            // Si es obligatorio, podríamos saltar o dejar que falle la validación.
            // La validación debería haberlo atrapado.
            $nCarton = 'SIN-CAJA';
        }

        // Crear o actualizar Artículo
        $this->createArticleIfNotExists([
            'mocaco' => $mocaco,
            'campana' => $campana,
            'seccion' => $this->normalizeValue($normalizedRow['seccion'] ?? null),
            'familia' => $this->normalizeValue($familiaRaw),
            'cadena' => $this->normalizeValue($normalizedRow['cadena'] ?? null),
            'mercado_origen_articulo' => $this->normalizeValue($mercadoRaw),
            'precio_pvp_maximo_temporada' => $precio,
            'partida_arancelaria' => $this->normalizeValue($partidaRaw),
            'composition' => $this->normalizeValue($composicionRaw),
            'peso_unitario' => $peso,
            'grupo_arancelario' => $this->normalizeValue($grupoRaw),
        ]);

        $models = [];

        // Atributos extraídos para uso repetido
        // OBLIGATORIO: Contenedor (prioridad Excel)
        $conteneur = $this->normalizeValue($normalizedRow['contenedor'] ?? $normalizedRow['conteneur'] ?? $this->containerEntry->n_camion);

        // OBLIGATORIO: Familia Usuario
        $familiaUsuario = $this->normalizeValue($normalizedRow['familia_usuario'] ?? $normalizedRow['famillie_usuario'] ?? $normalizedRow['famillie_u'] ?? $normalizedRow['familia'] ?? null);

        // OBLIGATORIO: Detail Usuario
        $detailUsuario = $this->normalizeValue($normalizedRow['detail_usuario'] ?? $normalizedRow['detail_usu'] ?? $normalizedRow['detalle_usuario'] ?? $normalizedRow['detalle'] ?? null);

        // OBLIGATORIO: Seccion
        $seccion = $this->normalizeValue($normalizedRow['seccion'] ?? null);

        // OBLIGATORIO: Cadena
        $cadena = $this->normalizeValue($normalizedRow['cadena'] ?? null);

        // OBLIGATORIO: Categoria (CATEG)
        $categoria = $this->normalizeValue($normalizedRow['categ'] ?? $normalizedRow['categoria'] ?? $normalizedRow['categoria_seleccionada'] ?? null);

        // Opcionales solicitados
        $seasonInt = $this->normalizeValue($normalizedRow['season_int'] ?? $normalizedRow['season'] ?? $normalizedRow['temporada'] ?? null);
        $notes = $this->normalizeValue($normalizedRow['notes'] ?? $normalizedRow['notas'] ?? 'Carga Masiva Excel');
        $ubicacion = $this->normalizeValue($normalizedRow['ubicacion'] ?? 'RECEPCION');

        for ($i = 0; $i < $cantidad; $i++) {
            // Generar full_barcode
            $fullBarcode = $this->normalizeValue($normalizedRow['full_barcode'] ?? $normalizedRow['barcode'] ?? $normalizedRow['codigo_barras'] ?? $mocaco);

            $models[] = new Inventory([
                'full_barcode' => $fullBarcode,
                'mocaco' => $mocaco,
                'n_carton' => $nCarton,
                'season_int' => $seasonInt,
                'notes' => $notes,
                'conteneur' => $conteneur,
                'n_id_operario' => $this->normalizeValue($normalizedRow['n_id_operario'] ?? $normalizedRow['operario'] ?? 'SISTEMA'),
                'ubicacion' => $ubicacion,
                'categoria_seleccionada' => $categoria,
                'famillie_usuario' => $familiaUsuario,
                'detail_usuario' => $detailUsuario, // Ensure detail_usuario is mapped
                'seccion' => $seccion,
                'familia_articulo_description' => $this->normalizeValue($familiaRaw),
                'cadena' => $cadena,
                'mercado_origen_articulo' => $this->normalizeValue($mercadoRaw),
                'precio_pvp_maximo_temporada' => $precio,
                'partida_arancelaria' => $this->normalizeValue($partidaRaw),
                'composition' => $this->normalizeValue($composicionRaw),
                'campana' => $campana,
                'peso_unitario' => $peso,
                'grupo_arancelario' => $this->normalizeValue($grupoRaw),
                'fecha_escaneo' => now(),
                'status' => 'disponible',
            ]);

            $this->importedCount++;
        }

        return $models;
    }

    /**
     * @return array
     */
    public function rules(): array
    {
        return [
            'mocaco' => 'required',
            'cantidad_unidades' => 'required',
            'n_carton' => 'required',
            'familia_usuario' => 'required',
            'detail_usuario' => 'required',
            'seccion' => 'required',
            'cadena' => 'required',
            'campana' => 'required',
            'categ' => 'required',
            'contenedor' => 'required',
        ];
    }

    /**
     * @return int
     */
    public function batchSize(): int
    {
        return 500;
    }

    /**
     * @return int
     */
    public function chunkSize(): int
    {
        return 500;
    }

    /**
     * Crear el artículo en la tabla articles SOLO si no existe
     */
    protected function createArticleIfNotExists(array $articleData)
    {
        try {
            // Buscar si ya existe un artículo con la misma combinación MOCACO + CAMPANA
            $existingArticle = Article::where('mocaco', $articleData['mocaco'])
                ->where(function ($query) use ($articleData) {
                    if ($articleData['campana'] !== null && $articleData['campana'] !== '') {
                        $query->where('campana', $articleData['campana']);
                    } else {
                        $query->whereNull('campana');
                    }
                })
                ->first();

            if (!$existingArticle) {
                Article::create($articleData);
            }
        } catch (\Exception $e) {
            Log::warning('Error al crear artículo desde importación de inventario: ' . $e->getMessage(), [
                'mocaco' => $articleData['mocaco'] ?? null,
            ]);
        }
    }

    /**
     * Get import statistics
     */
    public function getStats(): array
    {
        return [
            'imported' => $this->importedCount,
            'skipped' => $this->skippedCount,
        ];
    }
}
