<?php

namespace App\Imports;

use App\Models\Inventory;
use App\Models\BoxReservationItem;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;

class ReservationBoxesImport implements ToCollection, WithHeadingRow
{
    protected $reservationId;
    protected $boxNumbers = [];
    protected $count = 0;
    protected $errors = [];

    public function __construct(int $reservationId)
    {
        $this->reservationId = $reservationId;
    }

    /**
     * Process the collection
     */
    public function collection(Collection $rows)
    {
        foreach ($rows as $row) {
            // Normalize column names
            $normalized = [];
            foreach ($row as $key => $value) {
                $normalized[strtolower(trim($key))] = $value;
            }

            // Get box number from various possible column names
            $boxNumber = $normalized['box_number']
                ?? $normalized['n_carton']
                ?? $normalized['caja']
                ?? $normalized['numero_caja']
                ?? null;

            // Check if box is selected (column SELECCIONAR with X or 1)
            $selected = $normalized['seleccionar']
                ?? $normalized['selected']
                ?? $normalized['select']
                ?? null;

            // Skip if not selected or no box number
            if (empty($boxNumber) || empty($selected)) {
                continue;
            }

            // Normalize selection value
            $selected = strtoupper(trim($selected));
            if (!in_array($selected, ['X', '1', 'SI', 'YES', 'TRUE'])) {
                continue;
            }

            // Find inventory item by box number
            $inventory = Inventory::where('n_carton', trim($boxNumber))
                ->where('reservation_status', 'available')
                ->first();

            if (!$inventory) {
                $this->errors[] = "Caja {$boxNumber} no encontrada o no disponible";
                Log::warning("Box not found or not available for reservation", [
                    'box_number' => $boxNumber,
                    'reservation_id' => $this->reservationId
                ]);
                continue;
            }

            // Create reservation item
            BoxReservationItem::create([
                'reservation_id' => $this->reservationId,
                'inventory_id' => $inventory->id,
                'box_number' => trim($boxNumber),
                'quantity' => $inventory->cantidad ?? 1,
                'status' => 'reserved',
            ]);

            $this->boxNumbers[] = trim($boxNumber);
            $this->count++;
        }
    }

    /**
     * Get imported box numbers
     */
    public function getBoxNumbers(): array
    {
        return $this->boxNumbers;
    }

    /**
     * Get count of imported boxes
     */
    public function getCount(): int
    {
        return $this->count;
    }

    /**
     * Get errors
     */
    public function getErrors(): array
    {
        return $this->errors;
    }
}
