<?php

namespace App\Imports;

use App\Models\TheoreticalArticle;
use App\Models\ContainerEntry;
use App\Models\Article;
use App\Models\Brand;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Maatwebsite\Excel\Concerns\WithCustomValueBinder;
use PhpOffice\PhpSpreadsheet\Cell\Cell;
use PhpOffice\PhpSpreadsheet\Cell\DataType;
use PhpOffice\PhpSpreadsheet\Cell\DefaultValueBinder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class TheoreticalArticleImport extends DefaultValueBinder implements ToCollection, WithHeadingRow, WithChunkReading, WithCustomValueBinder
{
    /**
     * Bind value to cell - preserve leading zeros for mocaco column
     */
    public function bindValue(Cell $cell, $value)
    {
        // Si es la columna mocaco (columna A después de headers), forzar como texto
        if ($cell->getColumn() === 'A' && $cell->getRow() > 1) {
            $cell->setValueExplicit($value, DataType::TYPE_STRING);
            return true;
        }

        // Para otras columnas, usar el comportamiento por defecto
        return parent::bindValue($cell, $value);
    }

    protected $containerEntryId;
    protected $importedCount = 0;
    protected $skippedCount = 0;
    protected $progressCallback = null;
    protected $totalProcessed = 0;

    public function __construct(int $containerEntryId)
    {
        $this->containerEntryId = $containerEntryId;
    }

    /**
     * Set progress callback for tracking
     */
    public function setProgressCallback(callable $callback)
    {
        $this->progressCallback = $callback;
    }

    /**
     * Process collection in chunks
     */
    public function collection(Collection $rows)
    {
        if ($rows->isEmpty()) {
            return;
        }

        // Normalize all rows first
        $normalizedRows = $rows->map(function ($row) {
            $normalized = [];
            foreach ($row as $key => $value) {
                $normalized[strtolower(trim($key))] = $value;
            }
            return $normalized;
        })->filter(function ($row) {
            // Filter out rows without mocaco
            $mocacoRaw = $row['mocaco'] ?? $row['codigo'] ?? '';
            $mocaco = trim($this->numberToString($mocacoRaw));
            return !empty($mocaco);
        });

        if ($normalizedRows->isEmpty()) {
            return;
        }

        // Extract all mocaco+campana combinations from this chunk
        $chunkKeys = $normalizedRows->map(function ($row) {
            $mocaco = trim($this->numberToString($row['mocaco'] ?? $row['codigo'] ?? ''));
            $campana = $this->normalizeValue($row['campana'] ?? null);
            return ['mocaco' => $mocaco, 'campana' => $campana];
        });

        // Bulk fetch existing theoretical articles for this container
        $existingTheoretical = $this->fetchExistingTheoretical($chunkKeys);

        // Bulk fetch existing articles
        $existingArticles = $this->fetchExistingArticles($chunkKeys);

        // Prepare batch inserts
        $theoreticalToInsert = [];
        $articlesToInsert = [];
        $processedKeys = [];

        foreach ($normalizedRows as $row) {
            $mocaco = trim($this->numberToString($row['mocaco'] ?? $row['codigo'] ?? ''));
            $campana = $this->normalizeValue($row['campana'] ?? null);
            $key = $mocaco . '|' . ($campana ?? '');

            // Skip if already processed in this chunk
            if (isset($processedKeys[$key])) {
                $this->skippedCount++;
                continue;
            }
            $processedKeys[$key] = true;

            // Skip if exists in database
            if (isset($existingTheoretical[$key])) {
                $this->skippedCount++;
                continue;
            }

            // Prepare theoretical article data
            $theoreticalData = $this->prepareTheoreticalData($row, $mocaco, $campana);
            $theoreticalToInsert[] = $theoreticalData;

            // Prepare article data if not exists
            if (!isset($existingArticles[$key])) {
                $articleData = $this->prepareArticleData($row, $mocaco, $campana);
                $articlesToInsert[] = $articleData;
                $existingArticles[$key] = true; // Mark as will be inserted
            }

            $this->importedCount++;
        }

        // Bulk insert articles first (to avoid foreign key issues if any)
        if (!empty($articlesToInsert)) {
            $this->bulkInsertArticles($articlesToInsert);
        }

        // Bulk insert theoretical articles
        if (!empty($theoreticalToInsert)) {
            $this->bulkInsertTheoretical($theoreticalToInsert);
        }

        // Update progress
        $this->totalProcessed += $normalizedRows->count();
        if ($this->progressCallback) {
            call_user_func($this->progressCallback, $this->totalProcessed, $this->totalProcessed);
        }
    }

    /**
     * Fetch existing theoretical articles in bulk
     */
    protected function fetchExistingTheoretical(Collection $keys)
    {
        $mocacos = $keys->pluck('mocaco')->unique()->values()->all();

        $existing = TheoreticalArticle::where('container_entry_id', $this->containerEntryId)
            ->whereIn('mocaco', $mocacos)
            ->get(['mocaco', 'campana'])
            ->mapWithKeys(function ($item) {
                $key = $item->mocaco . '|' . ($item->campana ?? '');
                return [$key => true];
            })
            ->all();

        return $existing;
    }

    /**
     * Fetch existing articles in bulk
     */
    protected function fetchExistingArticles(Collection $keys)
    {
        $mocacos = $keys->pluck('mocaco')->unique()->values()->all();

        $existing = Article::whereIn('mocaco', $mocacos)
            ->get(['mocaco', 'campana'])
            ->mapWithKeys(function ($item) {
                $key = $item->mocaco . '|' . ($item->campana ?? '');
                return [$key => true];
            })
            ->all();

        return $existing;
    }

    /**
     * Bulk insert articles using INSERT IGNORE
     */
    protected function bulkInsertArticles(array $articles)
    {
        if (empty($articles)) {
            return;
        }

        try {
            // Add timestamps
            $now = now();
            foreach ($articles as &$article) {
                $article['created_at'] = $now;
                $article['updated_at'] = $now;
            }

            // Use chunk to avoid max packet size issues
            foreach (array_chunk($articles, 1000) as $chunk) {
                DB::table('articles')->insertOrIgnore($chunk);
            }
        } catch (\Exception $e) {
            Log::warning('Error bulk inserting articles: ' . $e->getMessage());
        }
    }

    /**
     * Bulk insert theoretical articles
     */
    protected function bulkInsertTheoretical(array $theoretical)
    {
        if (empty($theoretical)) {
            return;
        }

        try {
            // Add timestamps
            $now = now();
            foreach ($theoretical as &$item) {
                $item['created_at'] = $now;
                $item['updated_at'] = $now;
            }

            // Use chunk to avoid max packet size issues
            foreach (array_chunk($theoretical, 1000) as $chunk) {
                DB::table('theoretical_articles')->insert($chunk);
            }
        } catch (\Exception $e) {
            Log::error('Error bulk inserting theoretical articles: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Prepare theoretical article data
     */
    protected function prepareTheoreticalData(array $row, string $mocaco, $campana): array
    {
        $precioRaw = $row['precio_pvp_maximo_temporada']
            ?? $row['precio_pv']
            ?? $row['precio']
            ?? null;
        $precio = $this->parseDecimalValue($precioRaw);

        $pesoRaw = $row['peso_unitario']
            ?? $row['peso_unit']
            ?? $row['peso']
            ?? null;
        $peso = $this->parseDecimalValue($pesoRaw);

        $familiaRaw = $row['familia_articulo_description']
            ?? $row['familia_ar']
            ?? $row['descripcion_familia']
            ?? $row['familia']
            ?? null;

        $mercadoRaw = $row['mercado_origen_articulo']
            ?? $row['mercado']
            ?? $row['mercado_origen']
            ?? null;

        $partidaRaw = $row['partida_arancelaria']
            ?? $row['partida_ar']
            ?? $row['partida']
            ?? null;

        $composicionRaw = $row['composition']
            ?? $row['compositi']
            ?? $row['composicion']
            ?? null;

        $grupoRaw = $row['grupo_arancelario']
            ?? $row['grupo_ara']
            ?? $row['grupo']
            ?? null;

        $cantidadRaw = $row['cantidad_unidades'] ?? $row['unidades'] ?? $row['cantidad'] ?? 1;
        $cantidad = is_numeric($cantidadRaw) ? (int)$cantidadRaw : 1;

        return [
            'container_entry_id' => $this->containerEntryId,
            'mocaco' => $mocaco,
            'cantidad_unidades' => $cantidad,
            'season_int' => $this->normalizeValue($row['season_int'] ?? $row['season_in'] ?? $row['temporada'] ?? null),
            'categoria_seleccionada' => $this->normalizeValue($row['categoria_seleccionada'] ?? $row['categoria'] ?? null),
            'famillie_usuario' => $this->normalizeValue($row['famillie_usuario'] ?? $row['famillie_u'] ?? $row['familia_usuario'] ?? $row['familia'] ?? null),
            'detail_usuario' => $this->normalizeValue($row['detail_usuario'] ?? $row['detail_usu'] ?? $row['detalle_usuario'] ?? $row['detalle'] ?? null),
            'seccion' => $this->normalizeValue($row['seccion'] ?? null),
            'familia_articulo_description' => $this->normalizeValue($familiaRaw),
            'cadena' => Brand::normalize($this->normalizeValue($row['cadena'] ?? null)),
            'mercado_origen_articulo' => $this->normalizeValue($mercadoRaw),
            'precio_pvp_maximo_temporada' => $precio,
            'partida_arancelaria' => $this->normalizeValue($partidaRaw),
            'composition' => $this->normalizeValue($composicionRaw),
            'campana' => $campana,
            'peso_unitario' => $peso,
            'grupo_arancelario' => $this->normalizeValue($grupoRaw),
            'notes' => $this->normalizeValue($row['notes'] ?? $row['notas'] ?? null),
        ];
    }

    /**
     * Prepare article data
     */
    protected function prepareArticleData(array $row, string $mocaco, $campana): array
    {
        $precioRaw = $row['precio_pvp_maximo_temporada']
            ?? $row['precio_pv']
            ?? $row['precio']
            ?? null;
        $precio = $this->parseDecimalValue($precioRaw);

        $pesoRaw = $row['peso_unitario']
            ?? $row['peso_unit']
            ?? $row['peso']
            ?? null;
        $peso = $this->parseDecimalValue($pesoRaw);

        $familiaRaw = $row['familia_articulo_description']
            ?? $row['familia_ar']
            ?? $row['descripcion_familia']
            ?? $row['familia']
            ?? null;

        $mercadoRaw = $row['mercado_origen_articulo']
            ?? $row['mercado']
            ?? $row['mercado_origen']
            ?? null;

        $partidaRaw = $row['partida_arancelaria']
            ?? $row['partida_ar']
            ?? $row['partida']
            ?? null;

        $composicionRaw = $row['composition']
            ?? $row['compositi']
            ?? $row['composicion']
            ?? null;

        $grupoRaw = $row['grupo_arancelario']
            ?? $row['grupo_ara']
            ?? $row['grupo']
            ?? null;

        return [
            'mocaco' => $mocaco,
            'campana' => $campana,
            'seccion' => $this->normalizeValue($row['seccion'] ?? null),
            'familia' => $this->normalizeValue($familiaRaw),
            'cadena' => Brand::normalize($this->normalizeValue($row['cadena'] ?? null)),
            'mercado_origen_articulo' => $this->normalizeValue($mercadoRaw),
            'precio_pvp_maximo_temporada' => $precio,
            'partida_arancelaria' => $this->normalizeValue($partidaRaw),
            'composition' => $this->normalizeValue($composicionRaw),
            'peso_unitario' => $peso,
            'grupo_arancelario' => $this->normalizeValue($grupoRaw),
        ];
    }

    /**
     * Normaliza un valor: convierte "sin dato" a null, maneja números grandes, etc.
     */
    protected function normalizeValue($value)
    {
        if ($value === null) {
            return null;
        }

        $str = trim((string)$value);

        // Convertir "sin dato" a null
        if (strtolower($str) === 'sin dato' || $str === '') {
            return null;
        }

        return $str;
    }

    /**
     * Convierte un número (que puede venir como número o string) a string sin perder precisión
     * IMPORTANTE: Preserva ceros a la izquierda para códigos como mocaco
     */
    protected function numberToString($value)
    {
        if ($value === null) {
            return '';
        }

        // Convertir a string primero para manejar cualquier tipo
        $str = trim((string)$value);

        // Si está vacío, retornar vacío
        if ($str === '') {
            return '';
        }

        // CRÍTICO: Si el string empieza con 0 y es numérico, es un código con ceros a la izquierda
        // NO convertir a número, retornar como string directamente
        if (preg_match('/^0\d+$/', $str)) {
            return $str;
        }

        // Si es numérico (puede venir como string numérico o como número)
        if (is_numeric($value) || is_numeric($str)) {
            // Convertir a float para manejar notación científica si existe
            $num = (float)$str;

            // Si es un entero (sin decimales), devolverlo sin decimales
            if ($num == floor($num)) {
                // Usar number_format para evitar notación científica en números grandes
                return number_format($num, 0, '.', '');
            }

            // Si tiene decimales, usar formato con decimales pero sin notación científica
            return rtrim(number_format($num, 10, '.', ''), '0');
        }

        return $str;
    }

    /**
     * Convierte un valor numérico con coma decimal a float
     */
    protected function parseDecimalValue($value)
    {
        if ($value === null || $value === '') {
            return null;
        }

        $str = trim((string)$value);

        // Convertir "sin dato" a null
        if (strtolower($str) === 'sin dato') {
            return null;
        }

        // Reemplazar coma por punto para decimales
        $str = str_replace(',', '.', $str);

        // Remover espacios
        $str = str_replace(' ', '', $str);

        if (is_numeric($str)) {
            return (float)$str;
        }

        return null;
    }

    /**
     * @return int
     */
    public function chunkSize(): int
    {
        return 5000; // Increased from 500 to 5000
    }

    /**
     * Get import statistics
     */
    public function getStats(): array
    {
        return [
            'imported' => $this->importedCount,
            'skipped' => $this->skippedCount,
        ];
    }
}
