<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Article extends Model
{
    use HasFactory;

    protected $fillable = [
        'mocaco',
        'seccion',
        'familia',
        'cadena',
        'mercado_origen_articulo',
        'precio_pvp_maximo_temporada',
        'partida_arancelaria',
        'composition',
        'campana',
        'peso_unitario',
        'grupo_arancelario',
    ];

    protected $casts = [
        'precio_pvp_maximo_temporada' => 'decimal:2',
        'peso_unitario' => 'decimal:3',
    ];

    /**
     * Get the inventory items for this article.
     */
    public function inventoryItems(): HasMany
    {
        return $this->hasMany(Inventory::class, 'mocaco', 'mocaco');
    }

    /**
     * Extract MOCACO from full barcode.
     */
    public static function extractMocacoFromBarcode(string $fullBarcode): string
    {
        // Si el código es muy corto, devolver el código completo
        if (strlen($fullBarcode) < 11) {
            return $fullBarcode;
        }

        // Extraer desde la posición 1 (segundo carácter) hasta obtener 10 caracteres
        return substr($fullBarcode, 1, 10);
    }

    /**
     * Extract color from full barcode.
     * Color is digits 9-11 (positions 8-10 in 0-indexed)
     * Example: 00020609800016 → color: 800
     */
    public static function extractColorFromBarcode(string $fullBarcode): ?string
    {
        // Verificar que el código tenga al menos 11 dígitos
        if (strlen($fullBarcode) < 11) {
            return null;
        }

        // Extraer posiciones 9-11 (índices 8-10)
        return substr($fullBarcode, 8, 3);
    }

    /**
     * Extract size from full barcode.
     * Size is digits 12-13 (positions 11-12 in 0-indexed)
     * Example: 00020609800016 → size: 01
     */
    public static function extractSizeFromBarcode(string $fullBarcode): ?string
    {
        // Verificar que el código tenga al menos 13 dígitos
        if (strlen($fullBarcode) < 13) {
            return null;
        }

        // Extraer posiciones 12-13 (índices 11-12)
        return substr($fullBarcode, 11, 2);
    }
}
