<?php

namespace App\Models;

use App\Enums\ContainerEntryStatus;
use App\Models\Brand;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class ContainerEntry extends Model
{
    use HasFactory;

    /**
     * The "booted" method of the model.
     */
    protected static function booted(): void
    {
        static::saving(function (ContainerEntry $containerEntry) {
            if ($containerEntry->isDirty('marca') && !empty($containerEntry->marca)) {
                $containerEntry->marca = Brand::normalize($containerEntry->marca, true);
            }
        });
    }

    protected $fillable = [
        'n_camion',
        'marca',
        'tipo_producto',
        'estado',
        'usuario_asignado_id',
        'fecha_inicio_proceso',
        'fecha_finalizacion',
        'notes',
    ];

    protected $casts = [
        'estado' => ContainerEntryStatus::class,
        'fecha_inicio_proceso' => 'datetime',
        'fecha_finalizacion' => 'datetime',
    ];

    /**
     * Get the user assigned to this container entry.
     */
    public function usuarioAsignado(): BelongsTo
    {
        return $this->belongsTo(User::class, 'usuario_asignado_id');
    }

    /**
     * Usuarios asignados (múltiples) vía pivote.
     */
    public function usuariosAsignados(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'container_entry_user')
            ->withPivot(['role', 'assigned_at'])
            ->withTimestamps();
    }

    /**
     * Get the theoretical articles for this container entry.
     */
    public function theoreticalArticles(): HasMany
    {
        return $this->hasMany(TheoreticalArticle::class);
    }

    /**
     * Get the audit logs for this container entry.
     */
    public function auditLogs(): MorphMany
    {
        return $this->morphMany(AuditLog::class, 'model');
    }

    /**
     * Scope for filtering by estado.
     */
    public function scopeEnProceso($query)
    {
        return $query->where('estado', ContainerEntryStatus::EN_PROCESO->value);
    }

    /**
     * Scope for filtering by estado finalizado.
     */
    public function scopeFinalizado($query)
    {
        return $query->where('estado', ContainerEntryStatus::FINALIZADO->value);
    }

    /**
     * Scope for filtering by container number.
     */
    public function scopeByContainer($query, string $nCamion)
    {
        return $query->where('n_camion', $nCamion);
    }

    /**
     * Mark as finalized.
     */
    public function finalizar()
    {
        $this->update([
            'estado' => ContainerEntryStatus::FINALIZADO,
            'fecha_finalizacion' => now(),
        ]);
    }
}
