<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;

class Inventory extends Model
{
    use HasFactory;

    protected $table = 'inventory';

    protected $fillable = [
        'full_barcode',
        'mocaco',
        'n_carton',
        'season_int',
        'notes',
        'conteneur',
        'n_id_operario',
        'ubicacion',
        'categoria_seleccionada',
        'famillie_usuario',
        'seccion',
        'familia_articulo_description',
        'cadena',
        'mercado_origen_articulo',
        'precio_pvp_maximo_temporada',
        'partida_arancelaria',
        'composition',
        'campana',
        'peso_unitario',
        'grupo_arancelario',
        'detail_usuario',
        'fecha_escaneo',
        'status',
        'order_id',
        'needs_review',
        'review_notes',
        'reviewed_at',
        'reviewed_by',
        'color',
        'size',
        'reservation_status',
        'reserved_for_client_id',
        'reservation_id',
    ];

    protected $casts = [
        'precio_pvp_maximo_temporada' => 'decimal:2',
        'peso_unitario' => 'decimal:3',
        'fecha_escaneo' => 'datetime',
        'reviewed_at' => 'datetime',
        'needs_review' => 'boolean',
    ];

    /**
     * Get the article that owns this inventory item.
     */
    public function article(): BelongsTo
    {
        return $this->belongsTo(Article::class, 'mocaco', 'mocaco');
    }

    /**
     * Get the order items for this inventory item.
     */
    public function orderItems(): HasMany
    {
        return $this->hasMany(OrderItem::class);
    }

    /**
     * Get the order that owns this inventory item.
     */
    public function order(): BelongsTo
    {
        return $this->belongsTo(Order::class);
    }

    /**
     * Get the audit logs for this inventory item.
     */
    public function auditLogs(): MorphMany
    {
        return $this->morphMany(AuditLog::class, 'model');
    }

    /**
     * Get the reservation that owns this inventory item
     */
    public function reservation(): BelongsTo
    {
        return $this->belongsTo(BoxReservation::class, 'reservation_id');
    }

    /**
     * Get the client this item is reserved for
     */
    public function reservedForClient(): BelongsTo
    {
        return $this->belongsTo(Client::class, 'reserved_for_client_id');
    }

    /**
     * Check if item is available (not reserved)
     */
    public function isAvailable(): bool
    {
        return $this->reservation_status === 'available';
    }

    /**
     * Check if item is reserved
     */
    public function isReserved(): bool
    {
        return $this->reservation_status === 'reserved';
    }

    /**
     * Check if item reservation is confirmed
     */
    public function isReservationConfirmed(): bool
    {
        return $this->reservation_status === 'confirmed';
    }

    /**
     * Scope: Only available items
     */
    public function scopeAvailable($query)
    {
        return $query->where('reservation_status', 'available');
    }

    /**
     * Scope: Only reserved items
     */
    public function scopeReserved($query)
    {
        return $query->where('reservation_status', 'reserved');
    }

    /**
     * Scope for filtering by box number.
     * Matches case-insensitive with whitespace handling
     */
    public function scopeByBox($query, string $boxNumber)
    {
        $trimmedBoxNumber = trim($boxNumber);
        // Use LOWER for case-insensitive matching
        return $query->whereRaw('LOWER(TRIM(n_carton)) = LOWER(?)', [$trimmedBoxNumber]);
    }

    public function scopeByMocaco($query, string $mocaco)
    {
        $trimmedMocaco = trim($mocaco);
        // Use LOWER for case-insensitive matching
        return $query->whereRaw('LOWER(TRIM(mocaco)) = LOWER(?)', [$trimmedMocaco]);
    }

    /**
     * Scope for filtering by container.
     * Matches case-insensitive with whitespace handling
     */
    public function scopeByContainer($query, string $container)
    {
        $trimmedContainer = trim($container);
        // Use LOWER for case-insensitive matching
        return $query->whereRaw('LOWER(TRIM(conteneur)) = LOWER(?)', [$trimmedContainer]);
    }

    /**
     * Scope for filtering by barcode.
     */
    public function scopeByBarcode($query, string $barcode)
    {
        return $query->where('full_barcode', 'like', "%{$barcode}%");
    }
}
