<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;

class Order extends Model
{
    use HasFactory;

    protected $fillable = [
        'client_id',
        'user_id',
        'operario_id',
        'notes',
        'status',
        'order_date',
        'dispatched_date',
        'url_pdf_factura',
    ];

    protected $casts = [
        'order_date' => 'datetime',
        'dispatched_date' => 'datetime',
    ];

    /**
     * Get the client that owns this order.
     */
    public function client(): BelongsTo
    {
        return $this->belongsTo(Client::class);
    }

    /**
     * Get the user that created this order.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get the order items for this order.
     */
    public function orderItems(): HasMany
    {
        return $this->hasMany(OrderItem::class);
    }

    /**
     * Get the audit logs for this order.
     */
    public function auditLogs(): MorphMany
    {
        return $this->morphMany(AuditLog::class, 'model');
    }

    /**
     * Scope for draft orders.
     */
    public function scopeDraft($query)
    {
        return $query->where('status', 'draft');
    }

    /**
     * Scope for pending orders.
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Scope for reserved orders.
     */
    public function scopeReserved($query)
    {
        return $query->where('status', 'reserved');
    }

    /**
     * Scope for dispatched orders.
     */
    public function scopeDispatched($query)
    {
        return $query->where('status', 'dispatched');
    }

    /**
     * Confirm the order and update status.
     */
    public function confirm(): void
    {
        $this->update(['status' => 'pending']);
    }

    /**
     * Dispatch the order and update status.
     */
    public function dispatch(): void
    {
        $this->update([
            'status' => 'dispatched',
            'dispatched_date' => now(),
        ]);
    }
}