<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ScannerAlert extends Model
{
    protected $fillable = [
        'type',
        'keywords',
        'description',
        'is_active',
        'user_id',
    ];

    protected $casts = [
        'is_active' => 'boolean',
    ];

    /**
     * Get the user that owns this alert (if user-specific).
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Get keywords as array.
     */
    public function getKeywordsArrayAttribute(): array
    {
        $keywords = trim($this->keywords);
        if (empty($keywords)) {
            return [];
        }
        
        // Si es JSON, decodificar
        $decoded = json_decode($keywords, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($decoded)) {
            return array_map('trim', $decoded);
        }
        
        // Si no, separar por comas
        return array_map('trim', explode(',', $keywords));
    }

    /**
     * Check if a text matches any of the alert keywords.
     */
    public function matches(string $text): bool
    {
        if (empty($text)) {
            return false;
        }

        $textLower = mb_strtolower(trim($text));
        $keywords = $this->keywords_array;

        foreach ($keywords as $keyword) {
            $keywordLower = mb_strtolower(trim($keyword));
            if (!empty($keywordLower) && str_contains($textLower, $keywordLower)) {
                return true;
            }
        }

        return false;
    }
}
