<?php

namespace App\Services;

use App\Models\Inventory;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\AuditLog;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;

class InventoryService
{
    /**
     * Process barcode scan and add to inventory.
     */
    public function processBarcodeScan(array $data): array
    {
        try {
            DB::beginTransaction();

            // Validate article exists
            $article = \App\Models\Article::where('mocaco', $data['mocaco'])->first();
            if (!$article) {
                throw new \Exception('Artículo no encontrado en la base de datos.');
            }

            // Create inventory record
            $inventory = Inventory::create([
                'full_barcode' => $data['full_barcode'],
                'mocaco' => $data['mocaco'],
                'n_carton' => $data['n_carton'],
                'season_int' => $data['season_int'],
                'conteneur' => $data['conteneur'],
                'categoria_seleccionada' => $data['categoria_seleccionada'],
                'famillie_usuario' => $data['famillie_usuario'],
                'detail_usuario' => $data['detail_usuario'] ?? null,
                'ubicacion' => $data['ubicacion'] ?? null,
                'notes' => $data['notes'] ?? null,
                'n_id_operario' => Auth::user()->operario_id ?? Auth::user()->name,
                'fecha_escaneo' => now(),
                // Copy article data
                'seccion' => $article->seccion,
                'familia_articulo_description' => $article->familia,
                'cadena' => $article->cadena,
                'mercado_origen_articulo' => $article->mercado_origen_articulo,
                'precio_pvp_maximo_temporada' => $article->precio_pvp_maximo_temporada,
                'partida_arancelaria' => $article->partida_arancelaria,
                'composition' => $article->composition,
                'campana' => $article->campana,
                'peso_unitario' => $article->peso_unitario,
                'grupo_arancelario' => $article->grupo_arancelario,
            ]);

            // Log the action
            AuditLog::logAction(
                'scan',
                $inventory,
                Auth::user(),
                [],
                $inventory->toArray(),
                "Artículo escaneado: {$data['full_barcode']}"
            );

            DB::commit();

            return [
                'success' => true,
                'message' => 'Artículo escaneado y guardado correctamente.',
                'inventory' => $inventory,
            ];

        } catch (\Exception $e) {
            DB::rollBack();
            
            return [
                'success' => false,
                'message' => 'Error al procesar el escaneo: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Get box content with validation.
     */
    public function getBoxContent(string $boxNumber, ?string $container = null): array
    {
        $query = Inventory::byBox($boxNumber);

        if ($container) {
            $query->byContainer($container);
        }

        $articles = $query->with('article')->get();

        return [
            'success' => true,
            'articles' => $articles,
            'total' => $articles->count(),
        ];
    }

    /**
     * Update inventory item with audit trail.
     */
    public function updateInventoryItem(int $id, string $column, $value): array
    {
        try {
            $inventory = Inventory::findOrFail($id);
            $oldValue = $inventory->getAttribute($column);
            
            $inventory->update([$column => $value]);

            // Log the action
            AuditLog::logAction(
                'update_inventory',
                $inventory,
                Auth::user(),
                [$column => $oldValue],
                [$column => $value],
                "Campo {$column} actualizado"
            );

            return [
                'success' => true,
                'message' => 'Artículo actualizado correctamente.',
                'inventory' => $inventory,
            ];

        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Error al actualizar el artículo: ' . $e->getMessage(),
            ];
        }
    }

    /**
     * Delete inventory item with audit trail.
     */
    public function deleteInventoryItem(int $id, string $secretCode): array
    {
        // Validate secret code
        if ($secretCode !== '1234') {
            return [
                'success' => false,
                'message' => 'Código secreto incorrecto.',
            ];
        }

        try {
            $inventory = Inventory::findOrFail($id);
            $inventoryData = $inventory->toArray();
            
            $inventory->delete();

            // Log the action
            AuditLog::logAction(
                'delete_inventory',
                $inventory,
                Auth::user(),
                $inventoryData,
                [],
                "Artículo eliminado del inventario"
            );

            return [
                'success' => true,
                'message' => 'Artículo eliminado correctamente.',
            ];

        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Error al eliminar el artículo: ' . $e->getMessage(),
            ];
        }
    }
}










