<?php

namespace App\Traits;

use Illuminate\Support\Facades\Auth;

trait AuthorizesResourceAccess
{
    /**
     * Obtener el nombre del módulo basado en el nombre del controlador
     */
    protected function getModuleName(): string
    {
        $className = class_basename($this);
        // Remover "Controller" del final
        $moduleName = str_replace('Controller', '', $className);
        // Convertir a minúsculas
        $moduleName = strtolower($moduleName);

        // Mapeo de nombres de controladores a nombres de módulos
        $mapping = [
            'inventory' => 'inventory',
            'article' => 'articles',
            'client' => 'clients',
            'order' => 'orders',
            'relabeling' => 'relabeling',
            'containerentry' => 'container-entries', // Para ContainerEntryController
            'location' => 'location',
            'clienthistory' => 'client-history',
            'productivityreport' => 'reports.productivity', // Para ProductivityReportController
            'orderpreparation' => 'order-preparation', // Para OrderPreparationController
            'stocksummary' => 'reports.stock', // Para StockSummaryController
            'brand' => 'brands',
        ];

        return $mapping[$moduleName] ?? $moduleName;
    }

    /**
     * Verificar permiso para una acción específica
     */
    protected function authorizeAction(string $action): void
    {
        $moduleName = $this->getModuleName();

        // Mapeo de acciones a nombres de permisos específicos
        $actionMapping = [
            'assign' => 'assign',
            'export' => 'export',
            'dispatch' => 'dispatch',
            'upload' => 'upload',
            'finalize' => 'finalize',
        ];

        // Si la acción tiene un mapeo específico, usarlo; si no, usar la acción directamente
        $permissionAction = $actionMapping[$action] ?? $action;
        $permission = "{$moduleName}.{$permissionAction}";

        // Si el usuario es admin, permitir todo
        if (Auth::user()->hasRole('admin')) {
            return;
        }

        // Verificar si el permiso existe, si no, crearlo automáticamente
        if (!Auth::user()->can($permission)) {
            // Intentar crear el permiso si no existe
            $this->ensurePermissionExists($permission);

            // Verificar nuevamente
            if (!Auth::user()->can($permission)) {
                abort(403, "No tienes permiso para {$action} {$moduleName}.");
            }
        }
    }

    /**
     * Asegurar que un permiso existe en la base de datos
     */
    protected function ensurePermissionExists(string $permission): void
    {
        try {
            $permissionModel = \Spatie\Permission\Models\Permission::firstOrCreate(
                ['name' => $permission],
                ['guard_name' => 'web']
            );
        } catch (\Exception $e) {
            // Si hay error, simplemente continuar
            \Log::warning("No se pudo crear el permiso automáticamente: {$permission}");
        }
    }

    /**
     * Mapeo de métodos HTTP a acciones de permisos
     */
    protected function getActionFromMethod(string $method): string
    {
        $mapping = [
            'index' => 'view',
            'show' => 'view',
            'create' => 'create',
            'store' => 'create',
            'edit' => 'update',
            'update' => 'update',
            'destroy' => 'delete',
            'delete' => 'delete',
        ];

        return $mapping[$method] ?? 'view';
    }
}
