<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

class AddMissingPermissionsSeeder extends Seeder
{
    /**
     * Run the database seeds.
     * 
     * Este seeder agrega todos los permisos faltantes para los módulos:
     * - location (Asignar Ubicación)
     * - client-history (Historial Clientes)
     * - reports/productivity (Reportes de Productividad)
     * - order-preparation (Preparación de Pedidos)
     * - container-entries (Entradas de Contenedores)
     * - articles.delete (Eliminar artículos)
     */
    public function run(): void
    {
        $this->command->info('Creando permisos faltantes...');

        // Permisos para Location (Asignar Ubicación)
        $locationPermissions = [
            'location.view' => 'Ver asignación de ubicaciones',
            'location.assign' => 'Asignar ubicaciones a cajas',
        ];

        // Permisos para Client History (Historial Clientes)
        $clientHistoryPermissions = [
            'client-history.view' => 'Ver historial de clientes',
            'client-history.export' => 'Exportar historial de clientes',
        ];

        // Permisos para Reports/Productivity (Reportes de Productividad)
        $reportsPermissions = [
            'reports.view' => 'Ver reportes',
            'reports.productivity.view' => 'Ver reportes de productividad',
            'reports.productivity.export' => 'Exportar reportes de productividad',
            'reports.stock.view' => 'Ver resumen de stock',
        ];

        // Permisos para Order Preparation (Preparación de Pedidos)
        $orderPreparationPermissions = [
            'order-preparation.view' => 'Ver preparación de pedidos',
            'order-preparation.create' => 'Crear pedidos desde preparación',
            'order-preparation.dispatch' => 'Despachar pedidos desde preparación',
        ];

        // Permisos para Container Entries (Entradas de Contenedores)
        $containerEntriesPermissions = [
            'container-entries.view' => 'Ver entradas de contenedores',
            'container-entries.create' => 'Crear entradas de contenedores',
            'container-entries.update' => 'Actualizar entradas de contenedores',
            'container-entries.delete' => 'Eliminar entradas de contenedores',
            'container-entries.assign' => 'Asignar usuarios a contenedores',
            'container-entries.upload' => 'Subir archivos teóricos a contenedores',
            'container-entries.finalize' => 'Finalizar contenedores',
            'container-entries.export' => 'Exportar datos de contenedores',
        ];

        // Permiso faltante para Articles
        $articlesPermissions = [
            'articles.delete' => 'Eliminar artículos',
        ];

        // Combinar todos los permisos
        $allPermissions = array_merge(
            $locationPermissions,
            $clientHistoryPermissions,
            $reportsPermissions,
            $orderPreparationPermissions,
            $containerEntriesPermissions,
            $articlesPermissions
        );

        // Crear todos los permisos
        foreach ($allPermissions as $name => $description) {
            $permission = Permission::firstOrCreate(
                ['name' => $name],
                ['guard_name' => 'web']
            );
            $this->command->info("Permiso '{$name}' creado/verificado.");
        }

        $this->command->info('Permisos creados exitosamente.');

        // Asignar permisos al rol admin (todos los permisos)
        $adminRole = Role::where('name', 'admin')->first();
        if ($adminRole) {
            // Sincronizar todos los permisos (esto asegura que tenga todos, incluso los nuevos)
            $allPermissions = Permission::all();
            $adminRole->syncPermissions($allPermissions);
            $this->command->info('Todos los permisos asignados al rol admin.');
        }

        // Asignar permisos al rol manager
        $managerRole = Role::where('name', 'manager')->first();
        if ($managerRole) {
            $managerPermissions = [
                'location.view', 'location.assign',
                'client-history.view', 'client-history.export',
                'reports.view', 'reports.productivity.view', 'reports.productivity.export',
                'reports.stock.view',
                'order-preparation.view', 'order-preparation.create', 'order-preparation.dispatch',
                'container-entries.view', 'container-entries.create', 'container-entries.update',
                'container-entries.assign', 'container-entries.upload', 'container-entries.finalize',
                'container-entries.export',
                'articles.delete',
            ];
            foreach ($managerPermissions as $permName) {
                $perm = Permission::where('name', $permName)->first();
                if ($perm && !$managerRole->hasPermissionTo($perm)) {
                    $managerRole->givePermissionTo($perm);
                }
            }
            $this->command->info('Permisos asignados al rol manager.');
        }

        // Asignar permisos al rol operator (solo lectura y acciones básicas)
        // IMPORTANTE: El operator NO debe tener:
        // - articles.* (ningún permiso de artículos)
        // - clients.create, clients.update, clients.delete (solo clients.view)
        // - inventory.delete (solo inventory.view, create, update, scan)
        // - orders.delete, orders.dispatch (solo orders.view, create, update)
        $operatorRole = Role::where('name', 'operator')->first();
        if ($operatorRole) {
            // Definir TODOS los permisos que el operator debe tener (sincronizar)
            $operatorPermissions = [
                // Permisos de inventario (sin delete)
                'inventory.view', 'inventory.create', 'inventory.update', 'inventory.scan',
                // Permisos de pedidos (sin delete ni dispatch)
                'orders.view', 'orders.create', 'orders.update',
                // Permisos de clientes (solo view)
                'clients.view',
                // Permisos de reetiquetado
                'relabeling.view', 'relabeling.create', 'relabeling.print',
                // Nuevos permisos
                'location.view', 'location.assign',
                'client-history.view',
                'reports.productivity.view',
                'reports.stock.view',
                'order-preparation.view', 'order-preparation.create',
                'container-entries.view',
            ];
            
            // Obtener los objetos Permission
            $permissionsToSync = [];
            foreach ($operatorPermissions as $permName) {
                $perm = Permission::where('name', $permName)->first();
                if ($perm) {
                    $permissionsToSync[] = $perm;
                }
            }
            
            // Sincronizar permisos (esto quita los que no están en la lista y agrega los que faltan)
            $operatorRole->syncPermissions($permissionsToSync);
            $this->command->info('Permisos sincronizados para el rol operator (solo los permitidos).');
        }

        // Asignar permisos al rol viewer (solo lectura)
        $viewerRole = Role::where('name', 'viewer')->first();
        if ($viewerRole) {
            $viewerPermissions = [
                'location.view',
                'client-history.view',
                'reports.productivity.view',
                'reports.stock.view',
                'order-preparation.view',
                'container-entries.view',
            ];
            foreach ($viewerPermissions as $permName) {
                $perm = Permission::where('name', $permName)->first();
                if ($perm && !$viewerRole->hasPermissionTo($perm)) {
                    $viewerRole->givePermissionTo($perm);
                }
            }
            $this->command->info('Permisos asignados al rol viewer.');
        }

        $this->command->info('Seeder completado exitosamente.');
    }
}

