<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Client;
use App\Models\Article;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use Illuminate\Support\Facades\Hash;

class InitialDataSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Create roles
        $roles = [
            'admin' => 'Administrador del sistema',
            'manager' => 'Gerente de inventario',
            'operator' => 'Operador de inventario',
            'scanner' => 'Operador de escaneo',
            'viewer' => 'Solo lectura',
            'facturacion' => 'Perfil de facturación',
        ];

        foreach ($roles as $name => $description) {
            Role::firstOrCreate(['name' => $name], ['guard_name' => 'web']);
        }

        // Create permissions
        $permissions = [
            'inventory.view' => 'Ver inventario',
            'inventory.create' => 'Crear artículos en inventario',
            'inventory.update' => 'Actualizar inventario',
            'inventory.delete' => 'Eliminar del inventario',
            'inventory.scan' => 'Escanear artículos',
            'orders.view' => 'Ver pedidos',
            'orders.create' => 'Crear pedidos',
            'orders.update' => 'Actualizar pedidos',
            'orders.dispatch' => 'Despachar pedidos',
            'orders.delete' => 'Eliminar pedidos',
            'orders.manage_invoices' => 'Gestionar facturas de pedidos',
            'clients.view' => 'Ver clientes',
            'clients.create' => 'Crear clientes',
            'clients.update' => 'Actualizar clientes',
            'clients.delete' => 'Eliminar clientes',
            'relabeling.view' => 'Ver reetiquetado',
            'relabeling.create' => 'Crear productos de reetiquetado',
            'relabeling.print' => 'Imprimir etiquetas',
            'audit.view' => 'Ver auditoría',
            'users.view' => 'Ver usuarios',
            'users.create' => 'Crear usuarios',
            'users.update' => 'Actualizar usuarios',
            'users.delete' => 'Eliminar usuarios',
        ];

        foreach ($permissions as $name => $description) {
            Permission::firstOrCreate(['name' => $name], ['guard_name' => 'web']);
        }

        // Assign permissions to roles
        $adminRole = Role::findByName('admin');
        $adminRole->givePermissionTo(Permission::all());

        $managerRole = Role::findByName('manager');
        $managerRole->givePermissionTo([
            'inventory.view', 'inventory.create', 'inventory.update', 'inventory.scan',
            'orders.view', 'orders.create', 'orders.update', 'orders.dispatch',
            'clients.view', 'clients.create', 'clients.update',
            'relabeling.view', 'relabeling.create', 'relabeling.print',
            'audit.view',
        ]);

        $operatorRole = Role::findByName('operator');
        $operatorRole->givePermissionTo([
            'inventory.view', 'inventory.create', 'inventory.update', 'inventory.scan',
            // NO tiene inventory.delete
            'orders.view', 'orders.create', 'orders.update',
            // NO tiene orders.delete ni orders.dispatch
            'clients.view',
            // NO tiene clients.create, clients.update, clients.delete
            'relabeling.view', 'relabeling.create', 'relabeling.print',
            // NO tiene acceso a articles.*
        ]);

        $scannerRole = Role::findByName('scanner');
        $scannerRole->givePermissionTo([
            'inventory.view', 'inventory.create', 'inventory.scan',
            'orders.view', 'orders.create',
            'relabeling.view', 'relabeling.print',
        ]);

        $viewerRole = Role::findByName('viewer');
        $viewerRole->givePermissionTo([
            'inventory.view',
            'orders.view',
            'clients.view',
            'relabeling.view',
        ]);

        $facturacionRole = Role::findByName('facturacion');
        $facturacionRole->givePermissionTo([
            'orders.view',
            'orders.manage_invoices',
            'clients.view',
        ]);

        // Create admin user
        $admin = User::firstOrCreate(
            ['email' => 'admin@inventory.com'],
            [
                'name' => 'Administrador',
                'password' => Hash::make('password'),
                'operario_id' => 'ADMIN001',
                'is_active' => true,
            ]
        );
        $admin->assignRole('admin');

        // Create sample users
        $users = [
            [
                'name' => 'Gerente de Inventario',
                'email' => 'manager@inventory.com',
                'operario_id' => 'MGR001',
                'role' => 'manager',
            ],
            [
                'name' => 'Operador Principal',
                'email' => 'operator@inventory.com',
                'operario_id' => 'OPR001',
                'role' => 'operator',
            ],
            [
                'name' => 'Escáner',
                'email' => 'scanner@inventory.com',
                'operario_id' => 'SCN001',
                'role' => 'scanner',
            ],
        ];

        foreach ($users as $userData) {
            $user = User::firstOrCreate(
                ['email' => $userData['email']],
                [
                    'name' => $userData['name'],
                    'password' => Hash::make('password'),
                    'operario_id' => $userData['operario_id'],
                    'is_active' => true,
                ]
            );
            $user->assignRole($userData['role']);
        }

        // Create sample clients
        // $clients = [
        //     [
        //         'client_name' => 'Cliente Demo 1',
        //         'email' => 'juan@client1.com',
        //         'phone' => '+1234567890',
        //         'address' => 'Calle Principal 123, Ciudad',
        //     ],
        //     [
        //         'client_name' => 'Cliente Demo 2',
        //         'email' => 'maria@client2.com',
        //         'phone' => '+0987654321',
        //         'address' => 'Avenida Central 456, Ciudad',
        //     ],
        // ];

        // foreach ($clients as $clientData) {
        //     Client::firstOrCreate(
        //         ['client_name' => $clientData['client_name']],
        //         array_merge($clientData, ['is_active' => true])
        //     );
        // }

        // // Create sample articles
        // $articles = [
        //     [
        //         'mocaco' => '8574349707',
        //         'seccion' => 'SEC001',
        //         'familia' => 'Camisetas',
        //         'cadena' => 'ZARA',
        //         'mercado_origen_articulo' => 'España',
        //         'precio_pvp_maximo_temporada' => 29.99,
        //         'partida_arancelaria' => '61091000',
        //         'composition' => '100% Algodón',
        //         'campana' => 'Verano 2024',
        //         'peso_unitario' => 0.200,
        //         'grupo_arancelario' => 'GRP001',
        //     ],
        //     [
        //         'mocaco' => '8574349708',
        //         'seccion' => 'SEC002',
        //         'familia' => 'Pantalones',
        //         'cadena' => 'H&M',
        //         'mercado_origen_articulo' => 'Suecia',
        //         'precio_pvp_maximo_temporada' => 39.99,
        //         'partida_arancelaria' => '62034200',
        //         'composition' => '98% Algodón, 2% Elastano',
        //         'campana' => 'Verano 2024',
        //         'peso_unitario' => 0.350,
        //         'grupo_arancelario' => 'GRP002',
        //     ],
        // ];

        // foreach ($articles as $articleData) {
        //     Article::firstOrCreate(
        //         ['mocaco' => $articleData['mocaco']],
        //         $articleData
        //     );
        // }

        $this->command->info('Datos iniciales creados correctamente.');
        $this->command->info('Usuario admin: admin@inventory.com / password');
        $this->command->info('Usuario manager: manager@inventory.com / password');
        $this->command->info('Usuario operator: operator@inventory.com / password');
        $this->command->info('Usuario scanner: scanner@inventory.com / password');
    }
}