/**
 * Configuración de QZ Tray con firma digital
 * Este archivo debe cargarse después de qz-tray.js
 */

// Variable global para almacenar el certificado
let qzCertificate = null;

/**
 * Cargar el certificado público desde el servidor
 */
async function loadQZCertificate() {
    try {
        const response = await fetch('/digital-certificate.txt');
        if (!response.ok) {
            throw new Error('No se pudo cargar el certificado público');
        }
        qzCertificate = await response.text();
        console.log('Certificado QZ Tray cargado correctamente');
        return qzCertificate;
    } catch (error) {
        console.error('Error al cargar certificado QZ Tray:', error);
        throw error;
    }
}

/**
 * Configurar QZ Tray con la firma personalizada
 */
async function configureQZTray() {
    if (typeof qz === 'undefined') {
        console.error('QZ Tray no está cargado');
        return false;
    }

    try {
        // Cargar el certificado si no está cargado
        if (!qzCertificate) {
            await loadQZCertificate();
        }

        // Configurar QZ Tray para usar nuestro endpoint de firma
        qz.api.setShaType('sha512');
        
        // Configurar la función de firma personalizada
        // QZ Tray llamará a esta función cuando necesite firmar datos
        qz.security.setSignatureAlgorithm(function(dataToSign) {
            return new Promise(function(resolve, reject) {
                // Convertir los datos a string si es necesario
                let dataString = dataToSign;
                if (dataToSign instanceof ArrayBuffer) {
                    const bytes = new Uint8Array(dataToSign);
                    dataString = String.fromCharCode.apply(null, bytes);
                } else if (typeof dataToSign !== 'string') {
                    dataString = JSON.stringify(dataToSign);
                }

                fetch('/api/qz/sign', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'Accept': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                    },
                    body: JSON.stringify({ data: dataString })
                })
                .then(response => {
                    if (!response.ok) {
                        throw new Error('Error HTTP: ' + response.status);
                    }
                    return response.json();
                })
                .then(data => {
                    if (data.success && data.signature) {
                        // Convertir la firma de base64 a ArrayBuffer
                        const binaryString = atob(data.signature);
                        const bytes = new Uint8Array(binaryString.length);
                        for (let i = 0; i < binaryString.length; i++) {
                            bytes[i] = binaryString.charCodeAt(i);
                        }
                        resolve(bytes.buffer);
                    } else {
                        reject(new Error(data.error || 'Error al obtener la firma'));
                    }
                })
                .catch(error => {
                    console.error('Error en la firma:', error);
                    reject(error);
                });
            });
        });

        // Configurar el certificado público
        // QZ Tray necesita el certificado para verificar la firma
        if (qzCertificate) {
            qz.security.setCertificatePromise(function(resolve, reject) {
                if (qzCertificate) {
                    resolve(qzCertificate);
                } else {
                    reject(new Error('Certificado no cargado'));
                }
            });
        }

        console.log('QZ Tray configurado correctamente con firma personalizada');
        return true;
    } catch (error) {
        console.error('Error al configurar QZ Tray:', error);
        return false;
    }
}

/**
 * Conectar a QZ Tray con la configuración de firma
 */
async function connectQZTray() {
    try {
        // Configurar primero
        await configureQZTray();

        // Verificar si ya está conectado
        if (qz.websocket.isActive()) {
            console.log('QZ Tray ya está conectado');
            return true;
        }

        // Conectar
        await qz.api.connect();
        console.log('QZ Tray conectado exitosamente');
        return true;
    } catch (error) {
        console.error('Error al conectar QZ Tray:', error);
        throw error;
    }
}

