@extends('layouts.app')

@section('title', __('location.title'))

@section('content')
<div class="d-flex justify-content-between align-items-center mb-4">
    <h2><i class="fas fa-map-marker-alt"></i> {{ __('location.header') }}</h2>
</div>

<div class="row">
    <div class="col-md-8">
        <div class="card">
            <div class="card-header">
                <h5><i class="fas fa-info-circle"></i> {{ __('location.instructions.title') }}</h5>
            </div>
            <div class="card-body">
                <p>{{ __('location.instructions.text') }}</p>
            </div>
        </div>

        <div class="card mt-3">
            <div class="card-header">
                <h5><i class="fas fa-edit"></i> {{ __('location.form.title') }}</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <div class="form-group mb-3">
                            <label for="ubicacion-input">{{ __('location.form.ubicacion') }} <span class="text-danger">*</span></label>
                            <input type="text" id="ubicacion-input" class="form-control" placeholder="{{ __('location.form.ubicacion_placeholder') }}" required>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="form-group mb-3">
                            <label for="operario-input">{{ __('location.form.operario') }} <span class="text-danger">*</span></label>
                            <input type="text" id="operario-input" class="form-control" required>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="card mt-3">
            <div class="card-header">
                <h5><i class="fas fa-qrcode"></i> {{ __('location.scanning.title') }}</h5>
            </div>
            <div class="card-body">
                <div class="form-group mb-3">
                    <label for="barcode-caja-input">{{ __('location.scanning.label') }}</label>
                    <input type="text" id="barcode-caja-input" class="form-control form-control-lg" autofocus>
                </div>
                <div id="message" class="alert" style="display:none;"></div>
            </div>
        </div>

        <div class="card mt-3">
            <div class="card-header">
                <h5><i class="fas fa-list"></i> {{ __('location.scanning.scanned_title') }}</h5>
            </div>
            <div class="card-body">
                <div id="scanned-boxes-list" class="box-list" style="max-height: 350px; overflow-y: auto; border: 1px solid #ddd; padding: 10px; background-color: #f9f9f9; border-radius: 5px;">
                    <p class="text-center text-muted">{{ __('location.scanning.empty_message') }}</p>
                </div>
            </div>
        </div>

        <div class="card mt-3">
            <div class="card-body text-center">
                <button id="confirm-button" class="btn btn-success" disabled>
                    <i class="fas fa-check"></i> {{ __('location.buttons.confirm') }}
                </button>
                <button id="reset-button" class="btn btn-secondary">
                    <i class="fas fa-eraser"></i> {{ __('location.buttons.reset') }}
                </button>
            </div>
        </div>
    </div>

    <div class="col-md-4">
        <div class="card">
            <div class="card-header">
                <h5><i class="fas fa-keyboard"></i> {{ __('location.shortcuts.title') }}</h5>
            </div>
            <div class="card-body">
                <div class="shortcut-item">
                    <kbd>Enter</kbd> - {{ __('location.shortcuts.enter') }}
                </div>
                <div class="shortcut-item">
                    <kbd>Esc</kbd> - {{ __('location.shortcuts.esc') }}
                </div>
            </div>
        </div>

        <div class="card mt-3">
            <div class="card-header">
                <h5><i class="fas fa-info-circle"></i> {{ __('location.info.title') }}</h5>
            </div>
            <div class="card-body">
                <p class="small">{{ __('location.info.text') }}</p>
            </div>
        </div>
    </div>
</div>
@endsection

@push('scripts')
<script>
// Inyectar traducciones en JavaScript
@php
$locationTranslations = [
    'box_already_scanned' => __('location.messages.box_already_scanned'),
    'box_added' => __('location.messages.box_added'),
    'ubicacion_required' => __('location.messages.ubicacion_required'),
    'operario_required' => __('location.messages.operario_required'),
    'no_boxes' => __('location.messages.no_boxes'),
    'communication_error' => __('location.messages.communication_error'),
    'empty_message' => __('location.scanning.empty_message'),
    'confirm' => __('location.buttons.confirm'),
    'processing' => __('location.buttons.processing'),
];
@endphp
const translations = @json($locationTranslations);

// Función helper para formatear mensajes con parámetros
function formatMessage(key, params = {}) {
    let message = translations[key] || key;
    Object.keys(params).forEach(param => {
        message = message.replace(`:${param}`, params[param]);
    });
    return message;
}

document.addEventListener('DOMContentLoaded', function() {
    const ubicacionInput = document.getElementById('ubicacion-input');
    const operarioInput = document.getElementById('operario-input');
    const barcodeCajaInput = document.getElementById('barcode-caja-input');
    const messageDiv = document.getElementById('message');
    const scannedBoxesList = document.getElementById('scanned-boxes-list');
    const confirmButton = document.getElementById('confirm-button');
    const resetButton = document.getElementById('reset-button');

    let scannedBoxes = new Set(); // Usamos un Set para almacenar cajas únicas

    function showMessage(msg, type) {
        messageDiv.textContent = msg;
        messageDiv.className = `alert alert-${type}`;
        messageDiv.style.display = 'block';
        setTimeout(() => {
            messageDiv.style.display = 'none';
        }, 3000);
    }

    function updateConfirmButtonState() {
        confirmButton.disabled = scannedBoxes.size === 0 || !ubicacionInput.value.trim() || !operarioInput.value.trim();
    }

    function addBoxToList(boxNumber) {
        if (scannedBoxes.has(boxNumber)) {
            showMessage(formatMessage('box_already_scanned', { box: boxNumber }), 'danger');
            return;
        }

        scannedBoxes.add(boxNumber);
        const div = document.createElement('div');
        div.className = 'box-item';
        div.style.borderBottom = '1px dashed #eee';
        div.style.padding = '8px 0';
        div.innerHTML = `<span>${boxNumber}</span>`;
        scannedBoxesList.prepend(div); // Añadir al principio para ver los últimos escaneados

        // Limpiar mensaje predeterminado si es el primer elemento
        if (scannedBoxes.size === 1) {
            scannedBoxesList.innerHTML = '';
            scannedBoxesList.appendChild(div);
        }

        showMessage(formatMessage('box_added', { box: boxNumber }), 'success');
        updateConfirmButtonState();
    }

    barcodeCajaInput.addEventListener('change', function() {
        const boxNumber = this.value.trim();
        if (boxNumber) {
            addBoxToList(boxNumber);
        }
        this.value = ''; // Limpiar el input para el siguiente escaneo
        this.focus();
    });

    confirmButton.addEventListener('click', async function() {
        const ubicacion = ubicacionInput.value.trim();
        const operario = operarioInput.value.trim();

        if (!ubicacion) {
            showMessage(translations.ubicacion_required, 'danger');
            ubicacionInput.focus();
            return;
        }
        if (!operario) {
            showMessage(translations.operario_required, 'danger');
            operarioInput.focus();
            return;
        }
        if (scannedBoxes.size === 0) {
            showMessage(translations.no_boxes, 'danger');
            barcodeCajaInput.focus();
            return;
        }

        confirmButton.disabled = true; // Deshabilitar para evitar envíos dobles
        confirmButton.innerHTML = `<i class="fas fa-spinner fa-spin"></i> ${translations.processing}`;

        const boxesArray = Array.from(scannedBoxes); // Convertir Set a Array

        try {
            const response = await fetch('/location/assign', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                },
                body: JSON.stringify({
                    ubicacion: ubicacion,
                    operario: operario,
                    cajas: boxesArray
                })
            });
            const data = await response.json();

            if (data.success) {
                showMessage(data.message, 'success');
                scannedBoxes.clear();
                scannedBoxesList.innerHTML = `<p class="text-center text-muted">${translations.empty_message}</p>`;
                ubicacionInput.value = '';
            } else {
                showMessage(data.message, 'danger');
            }
        } catch (error) {
            console.error('Error al enviar datos:', error);
            showMessage(translations.communication_error, 'danger');
        } finally {
            updateConfirmButtonState();
            confirmButton.innerHTML = `<i class="fas fa-check"></i> ${translations.confirm}`;
            barcodeCajaInput.focus();
        }
    });

    resetButton.addEventListener('click', function() {
        ubicacionInput.value = '';
        operarioInput.value = '{{ Auth::user()->name }}';
        scannedBoxes.clear();
        scannedBoxesList.innerHTML = `<p class="text-center text-muted">${translations.empty_message}</p>`;
        messageDiv.style.display = 'none';
        updateConfirmButtonState();
        ubicacionInput.focus();
    });

    updateConfirmButtonState();
    ubicacionInput.focus();
});
</script>
@endpush

@push('styles')
<style>
.box-list {
    min-height: 100px;
}

.box-item {
    transition: background-color 0.2s;
}

.box-item:hover {
    background-color: #f0f0f0;
}
</style>
@endpush
